$(document).ready(function() {
  startup();

  function startup() {
    setupCallbacks();
    restorePreviousValues();
    updateDropdownState();

    window.socketMessageReceived = function(msg) {
      if (msg[2] == 0x1D) {
        updateValuesUsing(parseCurrentInputBuffer(msg));
        // getCurrentInputMenuSettings();
        /// Bug in AVR where it returns stale state immediately after changing source
        /// Mantis #0021025
        setTimeout(getCurrentInputMenuSettings, 500);
      } else if (msg[2] == 0x20) {
        updateInputName(msg.slice(5, 15));
      } else if (msg[2] == 0x34) {
        updateRoomEQOptions(parseRoomEQNames(msg.slice(5, msg.length - 1)));
      } else if (msg[2] == 0x28) {
        if (msg[3] == 0x85) {
          /// Input config not ready yet, retry after 0.5s
          setTimeout(getCurrentInputMenuSettings, 500);
          return;
        }
        hideSpinner();
        updateValuesUsing(parseInputConfigBuffer(msg));
        updateDropdownState();
        updateLogic16Availability(getDeviceName());

        // If we have room eq, fetch the names.
        // 4 == 'Not Calculated'
        if (msg[20] !== 4) {
          window.webSocket.send(0x34, [0xF0]);
        }
      } else if (msg[2] == 0x2D) {
        updateLipSyncStateWithVideoInputs(msg.slice(5, 11));
      }
    };

    // Get Input Config data
    window.webSocket.send(0x28, [0xF0]);
    window.webSocket.send(0x1D, [0xF0]);
  }

  function updateLogic16Availability(deviceName) {
    if (deviceName === "JBL SYNTHESIS") { return; }
    $("#2ch-mode option[value='8']").hide();
    $("#mch-mode option[value='5']").hide();
  }

  function updateDropdownState() {
    updateInputTrimEnabled();
    updateAudioSourceOptions();
    updateLipSyncState();
  }

  function getCurrentInputMenuSettings() {
    window.webSocket.send(0x28, [0xF0]);
  }

  function showSpinner() {
    const html = `
    <div id="loading-indicator" class="fullscreen-overlay">
      <div class="spinner-center text-center">
        <div class="spinner-border text-info" role="status">
          <span class="sr-only">Loading...</span>
        </div>
        <div class="text-white">Loading state</div>
      </div>
    </div>
    `;
    $("body").append(html).fadeIn(200);
  }

  function hideSpinner() {
    $("#loading-indicator").fadeOut(200).remove();
  }

  function setupCallbacks() {
    $("#source").change(function() {
      updateInputTrimEnabled();
      updateAudioSourceOptions();
      updateNameConfigurable();
      showSpinner();

      // Send new source ir code
      const newSource = getCodeForSource($(this).val());
      if (newSource) {
        window.webSocket.send(0x08, [0x10, newSource]);
      }
    });

    $("#audio-source").change(function() {
      updateInputTrimEnabled();
      updateAudioSourceOptions();
    });

    $("#input-name").inputFilter(function(value) {
      /// Match ascii characters
      return /^[\x00-\x7F]*$/.test(value);
    });

    var $inputName = $("#input-name");
    $inputName.on('input', function() {
      var remaining = $(this).val().length;
      var string = remaining + "/10";
      $("#input-name-remaining-characters").text(string);
    });

    $inputName.on('change', function() {
      var model = updateModel("input-name", $(this).val());
      sendModel(model);
    });

    // Disable return key form submission
    $inputName.keypress(function(event) {
      if (event.which == '13') {
        $inputName.change();
        event.preventDefault();
      }
    });
  }

  function getCodeForSource(newSource) {
    switch (newSource) {
      case "1": return 118;  // CD
      case "2": return 98;   // BD
      case "3": return 94;   // AV
      case "4": return 27;   // SAT
      case "5": return 96;   // PVR
      case "6": return 125;  // UHD
      case "8": return 99;   // AUX
      case "9": return 58;   // Display
      case "11": return 28;  // FM
      case "12": return 72;  // DAB
      case "14": return 92;  // NET
      case "16": return 100; // STB
      case "17": return 97;  // GAME
      case "18": return 122; // BT
      default: return null;
    }
  }

  function parseCurrentInputBuffer(buffer) {
    if (buffer.length != 7) {
      throw "Invalid input buffer";
    }

    if (!validateCommandCode(buffer, 0x1D)) {
      throw "Invalid input command code";
    }

    return {
      "source": buffer[5]
    };
  }

  function parseInputConfigBuffer(buffer) {
    if (buffer.length != 31) {
      return;
    }

    if (!validateCommandCode(buffer, 0x28)) {
      return;
    }

    var data = buffer.slice(5, buffer.length - 1);

    return {
      "input-name": Utils.asciiFromBuffer(data, { min: 0, max: 10 }),
      "lip-sync": data[10],
      "2ch-mode": data[11],
      "mch-mode": data[12],
      "bass": Utils.parseSignedValue(data[13]),
      "treble": Utils.parseSignedValue(data[14]),
      "room-eq": data[15],
      "input-trim": data[16],
      "dolby-audio": data[17],
      "stereo-mode": data[18],
      "sub-stereo": Utils.parseVolumeLevel(data[19]),
      "imax-mode": data[20],
      "auro-matic": data[21],
      "auro-matic-strength": data[22],
      "audio-source": data[23],
      "cd-direct": data[24]
    };
  }

  window.encodeModel = function(model) {
    var data = new Array(25).fill(32);
    const inputName = Utils.asciiToHex(model["input-name"]);
    data.splice(0, inputName.length, ...inputName);
    data[10] = model["lip-sync"];
    data[11] = model["2ch-mode"];
    data[12] = model["mch-mode"];
    data[13] = Utils.toSignedValue(model["bass"]);
    data[14] = Utils.toSignedValue(model["treble"]);
    data[15] = model["room-eq"];
    data[16] = model["input-trim"];
    data[17] = model["dolby-audio"];
    data[18] = model["stereo-mode"];
    data[19] = Utils.encodeVolumeLevel(model["sub-stereo"]);
    data[20] = model["imax-mode"];
    data[21] = model["auro-matic"];
    data[22] = model["auro-matic-strength"];
    data[23] = model["audio-source"];
    data[24] = model["cd-direct"];
    return { "code": 0x28, "data": data };
  };

  function updateInputTrimEnabled() {
    var currInput = $("#source :selected").text();
    var isDisabledAudioSource = $("#audio-source :selected").text() != "Analogue";
    var isDisabled = isVirtualInput(currInput) || isDisabledAudioSource;
    var inputTrim = $("#input-trim");
    if (isDisabled) {
      inputTrim.html('<option value="3">---</option>');
    } else {
      var html = `
        <option value="0">1V RMS</option>
        <option value="1">2V RMS</option>
        <option value="2">4V RMS</option>
      `;
      inputTrim.html(html);
      var selected = JSON.parse(sessionStorage.getItem("input-config"))["input-trim"];
      inputTrim.val(selected);
    }
    inputTrim.prop("disabled", isDisabled);
  }

  function updateAudioSourceOptions() {
    // Reset visibility of options
    $("#audio-source > option").each(function() {
      $(this).attr('disabled', false);
    });

    var currInput = $("#source :selected").text();

    // Hide Digital option for GAME input
    if (currInput == "GAME") {
      $("#audio-source option[value='1']").attr('disabled', true);
    }

    // Hide Analogue option for UHD and SAT inputs
    if (currInput == "UHD" || currInput == "SAT") {
      $("#audio-source option[value='0']").attr('disabled', true);
    }

    // Hide HDMI option for CD
    if (currInput == "CD") {
      $("#audio-source option[value='2']").attr('disabled', true);
    }

    // Disable picker for virtual inputs or Aux
    const isDisabled = isVirtualInput(currInput) || currInput == "AUX";
    $("#audio-source").prop("disabled", isDisabled);
  }

  function updateNameConfigurable() {
    var inputName = $("#source :selected").text();

    // Virtual inputs cannot be renamed
    $("#input-name").prop("disabled", isVirtualInput(inputName));
  }

  function isVirtualInput(inputName) {
    return inputName.match(/(fm|dab|net|display|bt)/i) != null;
  }

  function parseRoomEQNames(buffer) {
    if (buffer.length == 0) {
      return [];
    }

    if (buffer.length % 20 != 0) {
      console.log("Invalid room eq length (" + buffer.length + "), bailing..");
      return ["Not calculated"];
    }

    return chunkArray(buffer, 20)
      .map(x => Utils.asciiFromBuffer(x, { min: 0, max: 20 }));
  }

  function updateInputName(name) {
    if (name.length % 10 != 0) {
      console.log("Invalid input name (" + name.length + "), bailing..");
      return;
    }

    const newName = Utils.asciiFromBuffer(name, { min: 0, max: 20 });
    $("#input-name").val(newName);
    updateModel("input-name", newName);
  }

  function updateRoomEQOptions(names) {
    if (names.length === 0) {
      disableRoomEQ();
      return;
    }

    names.unshift("Off");

    const $roomEq = $("#room-eq");
    $roomEq.empty();
    $roomEq.prop('disabled', false);

    const options = names.map((element, i) => {
      const disabledState = element === "Not Calculated" ? " disabled" : "";
      return '<option value="' + i + '"' + disabledState + '>' + element + '</option>';
    });

    $roomEq.append(options);
    var model = JSON.parse(sessionStorage.getItem(window.currentPage));
    $roomEq.val(model["room-eq"]);
  }

  function disableRoomEQ() {
    const $roomEq = $("#room-eq");
    $roomEq.empty();
    $roomEq.prop('disabled', true);
    $roomEq.append('<option value="4">Not Calculated</option>');
  }

  function updateLipSyncState() {
    const currentInput = parseInt($("#source").val(), 10);
    var isDisabled = true;
    if (isHDMIInput(currentInput)) {
      isDisabled = false;
    } else {
      // Fetch current video input state, so that we can update the lipsync availability
      window.webSocket.send(0x2D, [0xF0]);
    }

    $("#lip-sync").prop('disabled', isDisabled);
  }

  function updateLipSyncStateWithVideoInputs(videoInputs) {
    const currentInput = parseInt($("#source").val(), 10);
    const index = videoInputByteForInput(currentInput);
    if (index == -1) { return; }
    // Check if current inputs video input is set to None.
    $("#lip-sync").prop('disabled', videoInputs[index] == 0x07);
  }

  function videoInputByteForInput(input) {
    return [1, 8, 11, 12, 14, 18].indexOf(input);
  }

  function isHDMIInput(currentInput) {
    return [2, 3, 4, 5, 6, 9, 16, 17].includes(currentInput);
  }
});
