#!/usr/bin/env python3

import os
import sys
import pickle
import aes_gcm
import hashlib
from pythonhkdf import hkdf

FULL_EFUSE_LEN = 512
A113_EFUSE_LEN = 256

def dhex(m, b):
    print(m, "".join("{:02x}".format(c) for c in b))
    #print(m, b.encode('hex'))

def get_ksecret(bl2aeskey):
    if len(bl2aeskey) != 32:
        return None
    salt = bl2aeskey[:16]
    ikm = bl2aeskey[16:]
    info = b"ksecretwrapping\x00"
    okm_len = 16

    kdf = hkdf.Hkdf(salt, ikm, hashlib.sha256)
    okm = kdf.expand(info, okm_len)

    return okm

if __name__ == '__main__':
    if len(sys.argv) != 4:
        print('invalid number of args')
        print('usage: %s <bl2aeskey> <pattern> <Epattern>' % sys.argv[0])
        exit()

    #init_value = int.from_bytes(os.urandom(12), 'big')
    init_value = int.from_bytes(b'\x10' * 12, 'big')
    auth_data = b''

    f = open(sys.argv[1], 'rb')
    bl2aeskey = f.read()

    f = open(sys.argv[2], 'rb')
    plaintext = f.read()

    if len(bl2aeskey) != 32:
        print('invalid key length', len(bl2aeskey))
        exit(1)

    if (len(plaintext) != FULL_EFUSE_LEN and
           len(plaintext) != A113_EFUSE_LEN):
        print('invalid input length', len(plaintext))
        exit(1)

    ksecret = get_ksecret(bl2aeskey)
    if len(ksecret) != 16:
        print('failed to get key')
        exit(1)

    dhex('ksecret', ksecret)

    gcm = aes_gcm.AES_GCM(int.from_bytes(ksecret, 'big'))
    encrypted, new_tag = gcm.encrypt(init_value, plaintext, auth_data)

    try:
        decrypted = gcm.decrypt(init_value, encrypted,
                new_tag + 1, auth_data)
    except aes_gcm.InvalidTagException:
        decrypted = gcm.decrypt(init_value, encrypted, new_tag, auth_data)

    fout = open(sys.argv[3], 'wb')
    fout.write(init_value.to_bytes(12, 'big'))
    fout.write(new_tag.to_bytes(16, 'big'))
    fout.write(encrypted)
