#! /bin/sh

# This script provides two functionalities:
# 1) mount/umount persistent ubi partition, if possible; no creation/recovery/..
# 2) format ubi volume (used for example for clearing settings partition)

PART=`cat /proc/mtd | grep '"data"' | sed -e 's/^mtd\([0-9]*\):.*/\1/'`
SUBPAGESIZE=`cat /sys/class/mtd/mtd${PART}/subpagesize`

SETTINGS_UBI_VOLUME="ubi0"
SETTINGS_VOLUME_NAME="nsdk-settings"
SETTINGS_PARTITION="${SETTINGS_UBI_VOLUME}:${SETTINGS_VOLUME_NAME}"
SETTINGS_MOUNTPOINT="/media/settings"

debug() {
	echo "[nsdk-ubi-mount]: $*"
}

ubi_mounted() {
	if mount | grep "${PARTITION}" > /dev/null; then
		return 0
	fi
	return 1
}

ubi_mount() {
	PARTITION="$1"
	local MOUNTPOINT="$2"

	if ubi_mounted; then
		debug "INFO: Ubi partition '${PARTITION}' already mounted"
	else
		if [ "x$PART" = "x" ]; then
			debug "ERROR: No such mtd partition 'data'"
			exit 1
		else
			mkdir -p "${MOUNTPOINT}"

			ubiattach /dev/ubi_ctrl -m "$PART" -O "$SUBPAGESIZE"
			mount -t ubifs -o ro "${PARTITION}" "${MOUNTPOINT}"

			if ubi_mounted; then
				debug "INFO: Ubi partition '${PARTITION}' mounted"
			else
				debug "ERROR: Unable to mount '${PARTITION}' partition"
				exit 1
			fi
		fi
	fi
}

ubi_unmount()
{
	PARTITION="$1"
	local MOUNTPOINT="$2"

	if ubi_mounted; then
		umount "${MOUNTPOINT}"
		ubidetach -m "$PART"

		if ubi_mounted; then
			debug "ERROR: Unable to unmount '${PARTITION}' partition"
			exit 1
		else
			debug "INFO: Ubi partition '${PARTITION}' unmounted"
		fi
	else
		debug "INFO: Ubi partition '${PARTITION}' is not mounted"
	fi
}

ubi_format() {
	local UBI_VOL="$1"
	local UBI_VOL_NAME="$2"
	PARTITION="${UBI_VOL}:${UBI_VOL_NAME}"

	if ubi_mounted; then
		debug "ERROR: Ubi partition '${PARTITION}' mounted"
		exit 1
	fi

	ubiattach /dev/ubi_ctrl -m "$PART" -O "$SUBPAGESIZE"

	ubirmvol /dev/${UBI_VOL} -N "${UBI_VOL_NAME}"
	ubimkvol /dev/${UBI_VOL} -m -N "${UBI_VOL_NAME}"

	ubidetach -m "$PART"
}

usage_exit() {
	debug "ERROR: Invalid args '$*'"
	exit 1
}

COMMAND="$1"

case "${COMMAND}" in
"mount")
	# partition, mountpoint
	[ "$#" -ne 3 ] && usage_exit "$*"
	ubi_mount "$2" "$3"
	;;
"mount_settings")
	[ "$#" -ne 1 ] && usage_exit "$*"
	ubi_mount "${SETTINGS_PARTITION}" "${SETTINGS_MOUNTPOINT}"
	;;
"unmount")
	# partition, mountpoint
	[ "$#" -ne 3 ] && usage_exit "$*"
	ubi_unmount "$2" "$3"
	;;
"unmount_settings")
	[ "$#" -ne 1 ] && usage_exit "$*"
	ubi_unmount "${SETTINGS_PARTITION}" "${SETTINGS_MOUNTPOINT}"
	;;
"format")
	# ubi volume, volume name
	[ "$#" -ne 3 ] && usage_exit "$*"
	ubi_format "$2" "$3"
	;;
"format_settings")
	[ "$#" -ne 1 ] && usage_exit "$*"
	ubi_format "${SETTINGS_UBI_VOLUME}" "${SETTINGS_VOLUME_NAME}"
	;;
*)
	debug "ERROR: Unknown command: ${COMMAND}"
	exit 1
	;;
esac
