$(document).ready(function () {
  const largeSpeakerValue = 0;
  const defaultSmallSpeakerValue = 5;

  registerCallbacks();
  startup();

  function registerCallbacks() {
    $("select").on("change", function() {
      valueChanged($(this).attr('id'));
    });
  }

  function valueChanged(identifier) {
    if (identifier === "front-left-right") {
      frontLeftRightValueChanged();
      surroundLeftRightValueChanged();
    } else if (identifier === "surround") {
      surroundLeftRightValueChanged();
    } else if (identifier === "height-front") {
      heightFrontValueChanged();
    } else if (identifier === "subwoofer") {
      subwooferValueChanged();
    } else if (identifier === "use-channels-6-7") {
      channelSixAndSevenValueChanged();
    } else if (identifier === "channel-13-14") {
      channel13And14ValueChanged();
    } else if (identifier === "channel-15-16") {
      channel15And16ValueChanged();
    }
    sendUpdatedModel();
  }

  function sendUpdatedModel() {
    var model = [];
    $('select').each(function() {
      var key = $(this).attr('id');
      var value = parseInt($(this).val(), 10);
      model[key] = value;
    });
    console.log(model);
    sessionStorage.setItem(window.currentPage, JSON.stringify(model));
    sendModel(model);
  }

  function startup() {
    restorePreviousValues();
    updateDropdownState();

    window.socketMessageReceived = function(msg) {
      if (msg[2] == 0x2A) {
        updateValuesUsing(parseSpeakerTypesBuffer(msg));
        updateDropdownState();
      } else if (msg[2] == 0x37) {
        parseRoomEQ(msg[5]);
      }
    };

    window.webSocket.send(0x2A, [0xF0]);
    window.webSocket.send(0x37, [0xF0]);
  }

  function parseSpeakerTypesBuffer(buffer) {
    if (buffer.length != 19) {
      throw "Invalid buffer length: " + buffer.length;
    }

    if (!validateCommandCode(buffer, 0x2A)) {
      throw "Invalid command code.";
    }

    var data = buffer.slice(5, buffer.length - 1);

    return {
      "front-left-right": data[0],
      "centre": data[1],
      "surround": data[2],
      "surround-back": data[3],
      "height-front": data[4],
      "height-back": data[5],
      "subwoofer": data[6],
      "channel-13-14": data[7],
      "channel-15-16": data[8],
      "height-type": data[9],
      "use-channels-6-7": data[10],
      "filter-slope": data[11],
      "sub-gain": data[12]
    };
  }

  function updateDropdownState() {
    frontLeftRightValueChanged();
    surroundLeftRightValueChanged();
    heightFrontValueChanged();
    channel13And14ValueChanged();
    channelSixAndSevenValueChanged();
  }

  // If Dirac is in use, you cannot change the speaker types.
  function parseRoomEQ(eq) {
    const isDisabled = (eq !== 0x00 && eq !== 0x04);
    if (isDisabled) {
      $("form.pt-5").hide();
      $("#dirac-speaker-types-warning").removeClass("d-none").addClass("d-flex");
    } else {
      $("form.pt-5").show();
      $("#dirac-speaker-types-warning").removeClass("d-flex").addClass("d-none");
    }
  }

  window.encodeModel = function(model) {
    var data = new Array(13);
    data[0]  = model["front-left-right"];
    data[1]  = model["centre"];
    data[2]  = model["surround"];
    data[3]  = model["surround-back"];
    data[4]  = model["height-front"];
    data[5]  = model["height-back"];
    data[6]  = model["subwoofer"];
    data[7]  = model["channel-13-14"];
    data[8]  = model["channel-15-16"];
    data[9]  = model["height-type"];
    data[10] = model["use-channels-6-7"];
    data[11] = model["filter-slope"];
    data[12] = model["sub-gain"];
    return { code: 0x2A, data: data };
  };

  function frontLeftRightValueChanged() {
    // Disable Large setting if front left/right speaker is small
    const isSmallFronts = !isLargeSpeaker($("#front-left-right").val());

    $("#centre, #surround, #surround-back")
      .each(function() {
        var currentValue = parseInt($(this).val(), 10);
        $(this).find("option[value='0']").prop("disabled", isSmallFronts);

        // If the current value was set to Large, reset to Small 80Hz
        if (currentValue == largeSpeakerValue && isSmallFronts) {
          $(this).val(defaultSmallSpeakerValue);
        }
      });

      // If front set to small, sub must be present
      if (!isSubwooferPresent() && isSmallFronts) {
        $("#subwoofer").val(0);
      }
  }

  function isSubwooferPresent() {
    return parseInt($("#subwoofer").val(), 10) === 0 ||
           parseInt($("#channel-13-14").val(), 10) === 16 ||
           parseInt($("#channel-15-16").val(), 10) === 32;
  }

  function surroundLeftRightValueChanged() {
    const noneSpeaker = 16;
    const isSurroundDisabled = parseInt($("#surround").val(), 10) == noneSpeaker;

    // Reset surround back value to none if surrounds are disabled
    if (isSurroundDisabled) {
      $("#surround-back").val(noneSpeaker);
    }

    $("#surround-back option[value!='16']").prop("disabled", isSurroundDisabled);

    const isFrontLRLarge = isLargeSpeaker($("#front-left-right").val());
    const isSurroundLarge = isLargeSpeaker($("#surround").val());
    const isSurrBackLarge = isLargeSpeaker($("#surround-back").val());
    const isLargeDisabled = !isFrontLRLarge || !isSurroundLarge;

    // If surround backs are set to large, and large is disabled, reset to default small.
    if (isSurrBackLarge && isLargeDisabled) {
      $("#surround-back").val(defaultSmallSpeakerValue);
    }

    $("#surround-back option[value='0']").prop("disabled", isLargeDisabled);
  }

  function isLargeSpeaker(value) {
    return parseInt(value, 10) == largeSpeakerValue;
  }

  function heightFrontValueChanged() {
    const heightFrontValue = parseInt($("#height-front").val(), 10);
    const $heightBack = $("#height-back");
    const noneSpeaker = 16;

    // Reset disabled options
    $heightBack.find("option").prop("disabled", false);

    const currentHeightBackValue = parseInt($heightBack.val(), 10);
    var filter;

    if (heightFrontValue == noneSpeaker) {
      setDefaultValueWithCondition(currentHeightBackValue != noneSpeaker, noneSpeaker);
      filter = function(value) { return value != noneSpeaker; };
    } else if (heightFrontValue == largeSpeakerValue) {
      // Height front is large, so back must be large or none
      const condition = currentHeightBackValue != noneSpeaker;
      setDefaultValueWithCondition(condition, largeSpeakerValue);

      // Disable small speakers
      filter = function(value) { return value > largeSpeakerValue && value < noneSpeaker; };
    } else {
      // Height front is small, so back must be small or none
      const condition = currentHeightBackValue == largeSpeakerValue;
      setDefaultValueWithCondition(condition, defaultSmallSpeakerValue);

      // Disable large speaker
      filter = function(value) { return value == largeSpeakerValue; };
    }

    // Disable invalid options
    $heightBack
      .find("option")
      .filter(filter)
      .prop("disabled", true);

    // Disable Height 2 if Height 1 is None
    $("#height-back").prop("disabled", heightFrontValue == noneSpeaker);
  }

  function setDefaultValueWithCondition(condition, defaultValue) {
    if (condition) {
      $("#height-back").val(defaultValue);
    }
  }

  function subwooferValueChanged() {
    resetFrontSpeakersIfNoSubsPresent();
  }

  function channel15And16ValueChanged() {
    resetFrontSpeakersIfNoSubsPresent();
  }

  function resetFrontSpeakersIfNoSubsPresent() {
    var $frontLR = $("#front-left-right");
    // If no sub present, and fronts are small, reset to large
    if (!isSubwooferPresent() && !isLargeSpeaker($frontLR.val())) {
      $frontLR.val(0);
    }
  }

  function channelSixAndSevenValueChanged() {
    var isDisabled = ["0", "3"].includes($("#use-channels-6-7").val());
    if (!isDisabled) {
      $("#surround-back").val("16");
    }
    $("#surround-back").prop("disabled", !isDisabled);
  }

  function channel13And14ValueChanged() {
    // If Channel 13 & 14 is Small, Channel 15 & 16 must be too
    var channelValue = parseInt($("#channel-13-14").val(), 10);
    var isDisabled = channelValue > 0 && channelValue < 16;
    var currentChannel1516Value = parseInt($("#channel-15-16").val(), 10);
    $("#channel-15-16 option[value='0']").prop("disabled", isDisabled);
    if (currentChannel1516Value == 0 && isDisabled) {
      $("#channel-15-16").val("6");
    }

    resetFrontSpeakersIfNoSubsPresent();
  }
});
