#!/usr/bin/env python3

import base64
import hashlib
import json
import os
import uuid
from cryptography.hazmat.backends import default_backend
from cryptography.hazmat.primitives import serialization
from masterKey_encrypt_key import MasterKeyEncryptor


def get_args():
	from argparse import ArgumentParser

	parser = ArgumentParser()
	parser.add_argument('--key', required=True, help='master key used for encryption (PEM format)')
	parser.add_argument('--settingsDirectory', required=True, help='root directory where settings to be encrypted are located')
	return parser.parse_args()


def maybe_encrypt_setting(setting_file, encryptor):
	# file might not be a setting, so ignore any deserialization errors
	with open(setting_file, 'r') as f:
		try:
			setting = json.load(f)
		except:
			return

	# we only care about encrypted settings which also have 'value'
	if not 'encrypted' in setting or not setting['encrypted'] or not 'value' in setting:
		return

	print('Encrypting setting file: ' + setting_file)

	# replace setting's 'value' with 'encryptedValue'
	setting_value = setting['value']

	setting_value_json = json.dumps(setting_value).encode('utf-8')
	setting_value_hash = hashlib.md5(setting_value_json).digest()

	setting['encryptedValue'] = 'OPT;' + base64.b64encode(encryptor.encrypt(setting_value_hash + setting_value_json)).decode('utf-8')
	del setting['value']

	with open(setting_file, 'w') as f:
		json.dump(setting, f, indent='\t')


def main():
	args = get_args()
	pub_key = None

	# encrypting key could be either a pubkey or privkey -- handle both
	try:
		with open(args.key, 'rb') as f:
			pub_key = serialization.load_pem_public_key(f.read(), backend=default_backend())
	except ValueError:
		with open(args.key, 'rb') as f:
			k = serialization.load_pem_private_key(f.read(), password=None, backend=default_backend())
			pub_key = k.public_key()

	e = MasterKeyEncryptor(pub_key)
	e.set_ta_uuid(uuid.UUID('84a205f9-fdc1-46cd-820c-0f9638e1e11e'))

	for root, subdirs, files in os.walk(args.settingsDirectory):
		for file in files:
			abs_filename = os.path.join(root, file)
			maybe_encrypt_setting(abs_filename, e)


if __name__ == "__main__":
	main()
