#!/bin/sh
set -u

# Called from udev; attempt to mount any added block devices and remove any
# removed devices.  Create a notify file for any audio CDs.

log_exit() {
	logger "mount.sh" "Exited due to an error!"
}
# To debug this script first uncomment the 'env > ' line below and then run
# this script with the mount-debug file
if [ -f "${1:-}" ]; then
	source "$1"
	set -x
else
	# Log a message if there is an unexpected error.
	trap log_exit EXIT
	# For debugging
	# env > /tmp/mount-debug
fi

MOUNT="/bin/mount"
APFSMOUNT="/usr/bin/apfs-fuse"
PMOUNT="/usr/bin/pmount"
UMOUNT="/bin/umount"
name="$(basename "$DEVNAME")"
audio_notify_link=/tmp/storage/notify-audio-cd


for line in $(grep -v '^#' /etc/udev/mount.blacklist); do
	if echo "$DEVNAME" | grep -q "$line"; then
		logger "udev/mount.sh" "[$DEVNAME] is blacklisted, ignoring"
		exit 0
	fi
done

# make sure the 'notifiacion' directory exists
mkdir -p /tmp/storage

isMounted() {
	grep -q "^$DEVNAME\s" /proc/mounts
}

automount() {
	! test -d "/media/$name" && mkdir -p "/media/$name"
	isMounted && return

	# iocharset/nls option works only for fat/ntfs; use nls option for ntfs to
	# avoid warning: Option iocharset is deprecated...
	case "${ID_FS_TYPE:-}" in
	vfat|fat|msdos)
		OPTION=",iocharset=utf8"
		;;
	ntfs)
		OPTION=",nls=utf8"
		;;
	ext2|ext3|ext4|hfs|hfsplus|iso9660|udf)
		OPTION=""
		;;
	apfs)
		OPTION=",allow_other"
		;;
	"")
		logger "mount.sh/automount" "udev passed empty filesystem type, continuing with auto type!"
		ID_FS_TYPE=auto
		OPTION=""
		;;
	*)
		logger "mount.sh/automount" "udev passed unknown filesystem type \"$ID_FS_TYPE\", continuing anyway!"
		OPTION=""
		;;
	esac

	if ( [ "${ID_FS_TYPE:-}" != "apfs" ] && ! $MOUNT -t "$ID_FS_TYPE" "$DEVNAME" "/media/$name" -o ro"$OPTION" ) || \
	   ( [ "${ID_FS_TYPE:-}" = "apfs" ] && ! $APFSMOUNT "$DEVNAME" "/media/$name" -o ro"$OPTION" ); then
		logger "mount.sh/automount" "$MOUNT -t $ID_FS_TYPE $DEVNAME \"/media/$name\" failed!"
		rmdir "/media/$name" || logger "mount.sh/automount" "Not removing non-empty directory /media/$name"
	else
		logger "mount.sh/automount" "Auto-mount of [/media/$name] successful"
		touch "/tmp/.automount-$name"

		# nSDK: create a symlink under /tmp/storage _after_ the actual mount
		# (but not for CD-ROMs).
		if [ -z "${ID_CDROM_MEDIA:-}" ]; then
			rm -f /tmp/storage/"$name"-*

			# Store the USB disk label in a separate file.
			echo "${ID_FS_LABEL_ENC:-}" > /tmp/storage/"$name"-"${ID_FS_UUID:-}".label
			ln -s /media/"$name" /tmp/storage/"$name"-"${ID_FS_UUID:-}"
		fi
	fi
}

autounmount() {
	logger "mount.sh/autounmount" "Unmounting device: $DEVNAME"
	# nSDK: remove the symlink under /tmp/storage (_before_ the actual umount!)
	rm -f /tmp/storage/"$name"-*

	for mnt in $(grep "^$DEVNAME\s" /proc/mounts | cut -f 2 -d " "); do
		$UMOUNT -l "$mnt"
	done

	# Remove empty directories from auto-mounter
	test -e "/tmp/.automount-$name" -a -d "/media/$name" && rmdir "/media/$name"
	logger "mount.sh/autounmount" "Umounted device: $DEVNAME"
}

if [ -n "$DEVNAME" -a -n "${ID_FS_TYPE:-}" -a \( \
		\( "${ID_BUS:-}" = "usb" -a "${ID_USB_DRIVER:-}" = "usb-storage" \) \
	-o \
		"${ID_BUS:-}" = "ata" \
	\) ]; then

	if [ -b "$DEVNAME" -a "${ACTION:-}" = add ]; then

		if [ -x "$PMOUNT" ]; then
			$PMOUNT "$DEVNAME" 2> /dev/null
		elif [ -x "$MOUNT" ]; then
			$MOUNT "$DEVNAME" 2> /dev/null
		fi

		# If the device isn't mounted at this point, it isn't configured in fstab
		# 20061107: Small correction: The rootfs partition may be called just "rootfs" and not by
		# 	    its true device name so this would break. If the rootfs is mounted on two places
		#	    during boot, it confuses the heck out of fsck. So Im auto-adding the root-partition
		#	    to /etc/udev/mount.blacklist via postinst
		automount
	fi

	if [ "${ACTION:-}" = remove ]; then
		autounmount
	fi
fi

# Handle CD for /dev/sr* devices.
if echo "$DEVNAME" | grep -q '^/dev/sr[0-9]$'; then
	if [ "${ACTION:-}" = add ]; then
		# When a CD drive is attached the nsdk-usb-storage-detect.sh script
		# creates a notify-scsi_host-host[0-9]+ entry in /tmp/storage.
		# At the time that script is run it's too early to determine what kind
		# of scsi host has been attached as the target LUNs haven't been probed
		# yet.  Therefore at this point _remove_ the notify entry, as it's now
		# clear that the device is a CD drive.
		#
		# Sample: DEVPATH=/devices/platform/ocp/47400000.usb/47401400.usb/musb-hdrc.0.auto/usb1/1-1/1-1:1.0/host14/target14:0:0/14:0:0:0/block/sr0
		#
		# Check if the scsi_host is there in the expected place.
		if [ -d "/sys$DEVPATH/../../../../scsi_host" ]; then
			# Extract the host[0-9]+ part:
			SHOST="$(ls /sys"$DEVPATH"/../../../../scsi_host/ | egrep 'host[0-9]+' | head -n 1)"
			if [ -L /tmp/storage/notify-scsi_host-"$SHOST" ]; then
				logger "mount.sh/handleCD" "Removing notify-scsi_host symlink for $SHOST"
				rm -f /tmp/storage/notify-scsi_host-"$SHOST"
			else
				logger "mount.sh/handleCD" "Couldn't find scsi_host for $DEVNAME"
			fi
		fi
	elif [ "${ACTION:-}" = change ]; then
		if [ "${ID_CDROM_MEDIA:-}" = 1 ]; then
			logger "mount.sh/handleCD" "Handling CD insertion"
			if [ -n "${ID_FS_TYPE:-}" -a -z "${ID_CDROM_MEDIA_TRACK_COUNT_AUDIO:-}" ]; then
				# This is a data CD-ROM, mount it normally using automount.
				automount
			else
				# This is an audio CD, create the symlink for the nsdk.
				logger "mount.sh/handleCD" "Creating audio cd notify link for $name"
				ln -s "/sys$DEVPATH" "$audio_notify_link-$name"
			fi
		else
			logger "mount.sh/handleCD" "Handling CD removal"
			rm -f "$audio_notify_link-$name"
			autounmount
		fi
	elif [ "${ACTION:-}" = remove ]; then
		# Special case for audio CDs; data CDs already have been removed by the
		# first autounmount.
		rm -f "$audio_notify_link-$name"
	fi
fi

# Remove exit trap.
trap "" EXIT
