#! /bin/sh

IFETH="eth0"
IFWLAN="wlan0"
IFAP="uap0"

TIMEOUT_ETH=10
TIMEOUT_WLAN=30

PROFILE="/tmp/network-profile"

debug()
{
	echo "| $@"
}

# jq returns empty if the file is not present, and null if the file is
# present but the requested value is missing.
is_null_or_empty()
{
	test "x$1" = "x" -o "x$1" = "xnull"
}

_jq()
{
	jq -rM "$1" $PROFILE
}


wait_link()
{
	local IF=$1
	local T=$2

	debug "Checking link on $IF..."

	for i in `seq 1 $T`; do
		if [ "x`cat /sys/class/net/$IF/carrier`" = "x1" ]; then return 0; fi
		debug "..$i"
		sleep 1
	done
	return 1
}

setup_ip()
{
	local IF=$1
	local NAV=$2
	debug "Setting up IP on $IF..."

	local DHCP=`_jq $NAV.dhcp`
	if is_null_or_empty "$DHCP" || [ "x$DHCP" != "xfalse" -a "x$DHCP" != "x0" ]; then
		debug "..use dhcp"
		# For flashing clean modules via TFTP we need some options from DHCP server:
		# default options:
		#   "boot_file" var -> dhcp_bootfile
		#   "ip" var -> dhcp_ip
		# extra requested options:
		#   17 - root path ("rootpath" var) -> dhcp_rootpath
		#   67 - vendor opt bootfile ("bootfile" var) -> dhcp_bootfile (only if not set via default option)
		#   150 - server ip ("opt150" var) -> dhcp_serverip
		# All of these dhcp_... variables are saved into /tmp/dhcpc-tftp file (via simple-networking-udhcpc.sh script)
		udhcpc -i $IF -O 17 -O 67 -O 150
	else
		local IP=`_jq $NAV.ip`
		local MASK=`_jq $NAV.netmask`
		local GW=`_jq $NAV.gateway`
		local DNS=`_jq "$NAV.dns[0]"`
		debug "..use static: $IP/$MASK, $GW, $DNS"

		ip ad add $IP/$MASK brd + dev $IF
		route add default gw $GW
		echo "nameserver $DNS" > /etc/resolv.conf

		# create fake /tmp/dhcpc-tftp file
		TFTP="/tmp/dhcpc-tftp"
		echo "dhcp_ip $IP" > $TFTP
		echo "dhcp_bootfile " >> $TFTP
		echo "dhcp_rootpath " >> $TFTP
		echo "dhcp_serverip " >> $TFTP
	fi
}

connect_wired()
{
	debug "Connect wired"

	setup_ip $IFETH '.networkProfile.wired'
}

wpa_supplicant_add_net_conf()
{
	local CONF="$1"
	local SSID="$2"
	local ENC="$3"
	local KEY="$4"
	local PRIORITY="$5"

	# prepare config file for wpa_supplicant
	echo "network={" >> $CONF
	echo "  ssid=\"$SSID\"" >> $CONF
	echo "  scan_ssid=1" >> $CONF
	echo "  priority=$PRIORITY" >> $CONF
	case "$ENC" in
	wpa_psk)
		echo "  ieee80211w=1" >> $CONF
		echo "  key_mgmt=WPA-PSK WPA-PSK-SHA256" >> $CONF
		if [ ${#KEY} -eq 64 ]; then
			echo "  psk=$KEY" >> $CONF
		else
			echo "  psk=\"$KEY\"" >> $CONF
		fi
		;;
	wep)
		echo "  key_mgmt=NONE" >> $CONF
		if [ ${#KEY} -eq 10 -o ${#KEY} -eq 26 ]; then
			echo "  wep_key0=$KEY" >> $CONF
		else
			echo "  wep_key0=\"$KEY\"" >> $CONF
		fi
		echo "  wep_tx_keyidx=0" >> $CONF
		;;
	*)
		echo "  key_mgmt=NONE" >> $CONF
	esac
	echo "}" >> $CONF
}

connect_wireless()
{
	local CONF="/tmp/wpa.conf"
	local NAV=".networkProfile.wirelessNetworks"
	local NSSIDS="`_jq "$NAV | length"`"

	NAV=".networkProfile.wireless"

	local ENC=`_jq $NAV.encryption`
	local SSID="`_jq $NAV.ssid`"
	local KEY="`_jq $NAV.key`"

	echo "" > $CONF

	if [ "x$ENC" != "xnull" -a "x$SSID" != "xnull" ]; then
		debug "Adding wireless $SSID to list of networks to be tried"

		wpa_supplicant_add_net_conf "$CONF" "$SSID" "$ENC" "$KEY" "$NSSIDS"

		# .networkProfile.wireless and .networkProfile.wirelessNetworks[0]
		# are identical no need to test twice, let's start the loop at [1].
		local IDX=1

		while [ "$IDX" -lt "$NSSIDS" ]; do
			local PRIORITY=$(( $NSSIDS - $IDX ))
			NAV=".networkProfile.wirelessNetworks[$IDX]"
			ENC=`_jq $NAV.encryption`
			SSID="`_jq $NAV.ssid`"
			KEY="`_jq $NAV.key`"

			if [ "x$ENC" != "xnull" -a "x$SSID" != "xnull" ]; then
				debug "Adding wireless $SSID to list of networks to be tried"

				wpa_supplicant_add_net_conf "$CONF" "$SSID" "$ENC" "$KEY" "$PRIORITY"
			fi

			IDX=$(( $IDX + 1 ))
		done

		wpa_supplicant -i$IFWLAN -c$CONF -B
		if wait_link $IFWLAN $TIMEOUT_WLAN; then
			setup_ip $IFWLAN $NAV
		else
			killall wpa_supplicant
			debug "No link, start SoftAP"
			connect_wireless_softap
		fi
	else
		debug "No profile, start SoftAP"
		connect_wireless_softap
	fi
}

connect_wireless_softap()
{
	local SSID="softap-$(cat /sys/class/net/${IFWLAN}/address | tr ':' '-')"
	local CHAN="2"
	local DHCPFROM="192.168.200.20"
	local DHCPTO="192.168.200.200"
	local DHCPNETMASK="255.255.255.0"

	local APADDR="192.168.200.1"
	local APNETMASK="255.255.255.0"

	debug "Start wireless_softap $SSID channel $CHAN"

	# make sure we have the $IFAP interface present
	[ -d /sys/class/net/$IFAP ] || iw phy phy0 interface add $IFAP type __ap

	# prepare hostapd.conf
	local CONF="/tmp/hostapd.conf"
	echo "interface=$IFAP" > $CONF
	echo "ssid=$SSID" >> $CONF
	echo "hw_mode=g" >> $CONF
	echo "channel=$CHAN" >> $CONF
	echo "macaddr_acl=0" >> $CONF
	echo "ieee80211n=1" >> $CONF

	# prepare udhcpd.conf
	local CONFDHCP="/tmp/udhcpd.conf"
	echo "interface $IFAP" > $CONFDHCP
	echo "start $DHCPFROM" >> $CONFDHCP
	echo "end $DHCPTO" >> $CONFDHCP
	echo "option subnet $DHCPNETMASK" >> $CONFDHCP
	echo "option domain local" >> $CONFDHCP
	echo "option lease 60 # 1 minute" >> $CONFDHCP

	# set local fixed IP
	ip link set dev $IFAP up
	ip ad add $APADDR/$APNETMASK dev $IFAP

	# start hostapd/udhcpd
	udhcpd $CONFDHCP
	hostapd -B $CONF
}

# load profile into the temporary file
simple-networking-get-profile.sh > $PROFILE

TYPE=`_jq '.networkProfile.type'`

# bring interfaces up
ip link set dev $IFETH up
ip link set dev $IFWLAN up


case "$TYPE" in
wired)
	debug "Force wired"
	connect_wired
	;;

wireless)
	debug "Force wireless"
	connect_wireless
	;;

# fallback to automatic in all other cases
*)
	# check wired link
	if [ -d /sys/class/net/$IFETH ] && wait_link $IFETH $TIMEOUT_ETH; then
		debug "Link ok -> use wired"
		connect_wired
	else
		debug "No link -> try wireless"
		connect_wireless
	fi
	;;

esac
