#!/bin/sh
#
# Record audio using mics, aux or spdif input and play it back

set -e

. /usr/share/factory-tool/scripts/factory_tool_include

DURATION="$1"
SOURCE="$2"
# Assign "all" value, if CHANNEL parameter is not provided
if [ "$3" = "" ]; then
	IN_CHANNEL="all"
else
	IN_CHANNEL="$3"
fi
if [ "$4" = "" ]; then
	OUT_CHANNEL="all"
else
	OUT_CHANNEL="$4"
fi

show_help()
{
cat << EOF
Usage:
	factory_tool test_audio_loopthrough <DURATION> <SOURCE> [IN_CHANNEL] [OUT_CHANNEL]

Description:
	Record audio from selected SOURCE and loop it directly to speakers.
	This test can be used during PCBA level testing, to check connection
	of full audio path.	Normally microphones are set to low sensitivity
	in order not to reach full saturation, nevertheless some calibration
	of artificial mouth and artificial ear is required to avoid positive
	feedback loop.

Parameters:
	DURATION
		Duration of recording (seconds).
	SOURCE = mics | aux | spdifin
		Source of recording. Could be microphones, AUX or SPDIF.
	IN_CHANNEL
		Which input channel to use.
		Value should be between 1 and 99 or all (default).
		Value shouldn't exceed maximum channel count.
	OUT_CHANNEL
		Which output channel to use.
		Value should be between 1 and 99 or all (default).
		Value shouldn't exceed maximum channel count.

Notes:
	For SPDIF 2 channel uncompressed audio should be used.
EOF
}

if [ "$1" = "help" ]; then
	show_help
	exit 0
elif [ $# -lt 2 ] || [ $# -gt 4 ]; then
	show_help
	exit 1
fi

check_channel "$OUT_CHANNEL" "$MAX_OUTPUT_CH"

case "$SOURCE" in
	mics)
		MAX_INPUT_CH="$MAX_MICS_INPUT_CH"
		;;
	aux)
		MAX_INPUT_CH="$MAX_AUX_INPUT_CH"
		;;
	spdifin)
		MAX_INPUT_CH=2
		;;
	*)
		echo "Error: wrong SOURCE parameter"
		show_help
		exit 1
		;;
esac

if [ `cat /etc/asound.conf | grep -q -i "pcm.$SOURCE"` ]; then
	echo "Error: SOURCE is not present"
	exit 1
fi

check_channel "$IN_CHANNEL" "$MAX_INPUT_CH"

MIX_MATRIX=`generate_channel_mix_matrix "$MAX_INPUT_CH" "$MAX_OUTPUT_CH" "$IN_CHANNEL" "$OUT_CHANNEL"`
gst-launch-1.0 alsasrc device="$SOURCE" ! \
	"audio/x-raw,format=S32LE,rate=48000,channels=$MAX_INPUT_CH" ! \
	audioconvert mix-matrix="$MIX_MATRIX" ! \
	"audio/x-raw,format=S32LE,rate=48000,channels=$MAX_OUTPUT_CH" ! \
	autoaudiosink &
sleep "$DURATION"
# Kill background precesses after sleep
pkill -P $$

exit 0
