#!/bin/bash -e

#set -x

# Path to aml_encrypt_gxl
TOOL_PATH=$(dirname $(readlink -f $0))

SCRIPT_PATH=$(dirname $(readlink -f $0))

# MARCO - START - Override paths
TOOL_PATH=$(dirname $(readlink -f $0))/tools
TMP=$(mktemp -d /tmp/aml-efuse-tool-XXXX)
# MARCO - STOP  - Override paths

# Temporary files directory
if [ -z "$TMP" ]; then
    TMP=${SCRIPT_PATH}/tmp
fi

if [ -z "$EXTERNAL_ENCRYPT_PATH" ]; then
    EXTERNAL_ENCRYPT_PATH=.
fi

if [ -z "$PRIVATE_KEY_PATH" ]; then
    PRIVATE_KEY_PATH=.
fi

usage() {
    cat << EOF
Usage: $(basename $0) --help
       $(basename $0) --generate-efuse-pattern \\
                      --soc [gxl | axg | txlx] \\
                      [--root-hash rootkeys.hash] \\
                      [--jtag-usb-password-hash password.hash] \\
                      [--scan-password-hash password.hash] \\
                      [--aes-key aeskey.bin] \\
                      [--enable-sb true] \\
                      [--enable-aes true] \\
                      [--enable-jtag-password true] \\
                      [--enable-usb-password true] \\
                      [--enable-scan-password true] \\
                      [--enable-anti-rollback true] \\
                      [--disable-boot-usb true] \\
                      [--disable-boot-spi true] \\
                      [--disable-boot-sdcard true] \\
                      [--disable-boot-nand-emmc true] \\
                      [--disable-boot-recover true] \\
                      [--disable-scan-chain true] \\
                      [--disable-print true] \\
                      [--disable-jtag true] \\
                      [--revoke-rsk-0 true] \\
                      [--revoke-rsk-1 true] \\
                      [--revoke-rsk-2 true] \\
                      [--revoke-rsk-3 true] \\
                      [--user-efuse-file <path-to-general-purpose-user-efuse-data>] \\
                      -o pattern.efuse

Environment Variables:
       TMP:          path to a temporary directory. Defaults to <this script's path>/tmp
EOF
    exit 1
}

check_file() {
    if [ ! -f "$2" ]; then echo Error: Unable to open $1: \""$2"\"; exit 1 ; fi
}

check_boolean() {
    if [ "$2" != "true" ] && [ "$2" != "false" ]; then
        echo Error: invalid value $1: \""$2"\"
        exit 1
    fi
}

# Pad file to len by adding 0's to end of file
# $1: file
# $2: len
pad_file() {
    local file=$1
    local len=$2
    if [ ! -f "$1" ] || [ -z "$2" ]; then
        echo "Argument error, \"$1\", \"$2\" "
        exit 1
    fi
    local filesize=$(wc -c < ${file})
    local padlen=$(( $len - $filesize ))
    if [ $len -lt $filesize ]; then
        echo "File larger than expected.  $filesize, $len"
        exit 1
    fi
    dd if=/dev/zero of=$file oflag=append conv=notrunc bs=1 \
        count=$padlen >& /dev/null
}

# $1: input
# $2: output
# $3: aes key file
# $4: aes iv file
internal_encrypt() {
    local input=$1
    local output=$2
    local keyfile=$3
    local ivfile=$4
    if [ ! -f "$1" ] || [ -z "$2" ] || [ -z "$3" ] || [ -z "$4" ]; then
        echo "Argument error"
        exit 1
    fi
    local key=$(xxd -p -c64 $keyfile)
    local iv=$(xxd -p -c64 $ivfile)
    local imagesize=$(wc -c < ${input})
    local rem=$(( $imagesize % 16 ))
    if [ $rem -ne 0 ]; then
        echo "Input $input not 16 byte aligned?"
        exit 1
    fi
    openssl enc -aes-256-cbc -K $key -iv $iv -e -in $input -out $output -nopad
}

# $1: input
# $2: output
external_encrypt() {
    local input=$1
    local output=$2
    if [ ! -f "$1" ] || [ -z "$2" ]; then
        echo "Argument error, \"$1\", \"$2\" "
        exit 1
    fi
    local imagesize=$(wc -c < ${input})
    local rem=$(( $imagesize % 16 ))
    if [ $rem -ne 0 ]; then
        echo "Input $input not 16 byte aligned?"
        exit 1
    fi

    echo Not implemented
    #cp $input $output
    exit 1
}

# Get key len in bytes of private PEM RSA key
# $1: PEM file
get_pem_key_len() {
    local pem=$1
    local bits=0
    if [ ! -f "$1" ]; then
        echo "Argument error, \"$1\""
        exit 1
    fi
    bits=$(openssl rsa -in $pem -text -noout | \
        grep '^Private-Key: (' | \
        sed 's/Private-Key: (//' | \
        sed 's/ bit)//')
    if [ "$bits" -ne 1024 ] && [ "$bits" -ne 2048 ] &&
       [ "$bits" -ne 4096 ] && [ "$bits" -ne 8192]; then
       echo "Unexpected key size  $bits"
       exit 1
    fi
    echo $(( $bits / 8 ))
}

roothash=""
jtagusbpasswordhash=""
scanpasswordhash=""
userefusefile=""
aeskey=""
aeskeyta=""
enablesb="false"
enableaes="false"
enablejtagpassword="false"
enableusbpassword="false"
enablescanpassword="false"
enableantirollback="false"
disablebootusb="false"
disablebootspi="false"
disablebootsdcard="false"
disablebootnandemmc="false"
disablebootrecover="false"
disableprint="false"
disablejtag="false"
disablescanchain="false"
revokersk0="false"
revokersk1="false"
revokersk2="false"
revokersk3="false"
output=""
sigver=""
keyhashver=""
soc="gxl"

generate_efuse_pattern() {
    local argv=("$@")
    local i=0

    # Parse args
    i=0
    while [ $i -lt $# ]; do
        arg="${argv[$i]}"
        i=$((i + 1))
        case "$arg" in
            --root-hash)
                roothash="${argv[$i]}" ;;
            --jtag-usb-password-hash)
                jtagusbpasswordhash="${argv[$i]}" ;;
            --scan-password-hash)
                scanpasswordhash="${argv[$i]}" ;;
            --aes-key)
                aeskey="${argv[$i]}" ;;
            --aes-key-ta)
                aeskeyta="${argv[$i]}" ;;
            --enable-sb)
                enablesb="${argv[$i]}" ;;
            --enable-aes)
                enableaes="${argv[$i]}" ;;
            --enable-jtag-password)
                enablejtagpassword="${argv[$i]}" ;;
            --enable-usb-password)
                enableusbpassword="${argv[$i]}" ;;
            --enable-scan-password)
                enablescanpassword="${argv[$i]}" ;;
            --enable-anti-rollback)
                enableantirollback="${argv[$i]}" ;;
            --disable-boot-usb)
                disablebootusb="${argv[$i]}" ;;
            --disable-boot-spi)
                disablebootspi="${argv[$i]}" ;;
            --disable-boot-sdcard)
                disablebootsdcard="${argv[$i]}" ;;
            --disable-boot-nand-emmc)
                disablebootnandemmc="${argv[$i]}" ;;
            --disable-boot-recover)
                disablebootrecover="${argv[$i]}" ;;
            --disable-print)
                disableprint="${argv[$i]}" ;;
            --disable-jtag)
                disablejtag="${argv[$i]}" ;;
            --disable-scan-chain)
                disablescanchain="${argv[$i]}" ;;
            --revoke-rsk-0)
                revokersk0="${argv[$i]}" ;;
            --revoke-rsk-1)
                revokersk1="${argv[$i]}" ;;
            --revoke-rsk-2)
                revokersk2="${argv[$i]}" ;;
            --revoke-rsk-3)
                revokersk3="${argv[$i]}" ;;
            --user-efuse-file)
                userefusefile="${argv[$i]}" ;;
            -o)
                output="${argv[$i]}" ;;
            --sig-ver)
                sigver="${argv[$i]}" ;;
            --key-hash-ver)
                keyhashver="${argv[$i]}" ;;
            --generate-efuse-pattern)
		i=$((i - 1)) ;;
            --soc)
                soc="${argv[$i]}" ;;
            *)
                echo "Unknown option $arg"; exit 1
                ;;
        esac
        i=$((i + 1))
    done

    # Verify args
    if [ -z "$output" ]; then echo Error: Missing output file option -o; exit 1; fi

    if [ -z "$sigver" ]; then
        sigver=1
    fi

    if [ -z "$keyhashver" ]; then
        keyhashver=1
    fi

    if [ "$soc" != "gxl" ] && [ "$soc" != "axg" ] && [ "$soc" != "txlx"]; then
        echo Error: invalid soc: \"$soc\"
	exit 1
    fi

    check_boolean enable-sb "$enablesb"
    check_boolean enable-aes "$enableaes"
    check_boolean enable-jtag-password "$enablejtagpassword"
    check_boolean enable-usb-password "$enableusbpassword"
    check_boolean enable-jtag-password "$enablescanpassword"
    check_boolean enable-anti-rollback "$enableantirollback"
    check_boolean disable-boot-usb "$disablebootusb"
    check_boolean disable-boot-spi "$disablebootspi"
    check_boolean disable-boot-sdcard "$disablebootsdcard"
    check_boolean disable-boot-nand-emmc "$disablebootnandemmc"
    check_boolean disable-boot-recover "$disablebootrecover"
    check_boolean disable-print "$disableprint"
    check_boolean disable-jtag "$disablejtag"
    check_boolean disable-scan-chain "$disablescanchain"
    check_boolean revoke-rsk-0 "$revokersk0"
    check_boolean revoke-rsk-1 "$revokersk1"
    check_boolean revoke-rsk-2 "$revokersk2"
    check_boolean revoke-rsk-3 "$revokersk3"

    if [ "$roothash" != "" ]; then
        check_file roothash "$roothash"
        # Check size
        local filesize=$(wc -c < ${roothash})
        if [ $filesize -ne 32 ]; then
            echo "Root key hash incorrect size $filesize != 32"
            exit 1
        fi
    fi
    if [ "$jtagusbpasswordhash" != "" ]; then
        check_file passwordhash "$jtagusbpasswordhash"
        # Check size
        local filesize=$(wc -c < ${jtagusbpasswordhash})
        if [ $filesize -ne 32 ]; then
            echo "Jtag/Usb password key hash incorrect size $filesize != 32"
            exit 1
        fi
    fi
    if [ "$scanpasswordhash" != "" ]; then
        check_file passwordhash "$scanpasswordhash"
        # Check size
        local filesize=$(wc -c < ${scanpasswordhash})
        if [ $filesize -ne 32 ]; then
            echo "Scan password key hash incorrect size $filesize != 32"
            exit 1
        fi
    fi
    if [ "$aeskey" != "" ]; then
        check_file aeskey "$aeskey"
        filesize=$(wc -c < ${aeskey})
        if [ $filesize -ne 32 ]; then
            echo "AES key incorrect size $filesize != 32"
            exit 1
        fi
    fi
    if [ "$aeskeyta" != "" ]; then
        check_file aeskeyta "$aeskeyta"
        filesize=$(wc -c < ${aeskeyta})
        if [ $filesize -ne 32 ]; then
            echo "AES key ta incorrect size $filesize != 32"
            exit 1
        fi
    fi

    if [ "$soc" == "gxl" ] || [ "$soc" == "txlx" ]; then
	generate_efuse_pattern_gxl
    else
	generate_efuse_pattern_axg
    fi
}

generate_efuse_pattern_gxl() {
    # Additional args check
    if [ "$userefusefile" != "" ]; then
        check_file userefusefile "$userefusefile"
        local filesize=$(wc -c < ${userefusefile})
        if [ $filesize -ne 192 ]; then
            echo "User efuse file incorrect size $filesize != 192"
            exit 1
        fi
    fi

    # Create input pattern
    b4="00"
    if [ "$disablebootusb" == "true" ]; then
       b4="$(printf %02x $(( 0x$b4 | 0x10 )))"
    fi
    if [ "$disablebootspi" == "true" ]; then
       b4="$(printf %02x $(( 0x$b4 | 0x20 )))"
    fi
    if [ "$disablebootsdcard" == "true" ]; then
       b4="$(printf %02x $(( 0x$b4 | 0x40 )))"
    fi
    if [ "$disablebootnandemmc" == "true" ]; then
       b4="$(printf %02x $(( 0x$b4 | 0x80 )))"
    fi

    b5="00"
    if [ "$disablebootrecover" == "true" ]; then
       b5="$(printf %02x $(( 0x$b5 | 0x01 )))"
    fi
    if [ "$disableprint" == "true" ]; then
       b5="$(printf %02x $(( 0x$b5 | 0x02 )))"
    fi
    if [ "$disablejtag" == "true" ]; then
       b5="$(printf %02x $(( 0x$b5 | 0x04 )))"
    fi
    if $revokersk0; then
       b5="$(printf %02x $(( 0x$b5 | 0x10 )))"
    fi
    if $revokersk1; then
       b5="$(printf %02x $(( 0x$b5 | 0x20 )))"
    fi
    if $revokersk2; then
       b5="$(printf %02x $(( 0x$b5 | 0x40 )))"
    fi
    if $revokersk3; then
       b5="$(printf %02x $(( 0x$b5 | 0x80 )))"
    fi

    b6="00"
    if [ "$enablesb" == "true" ]; then
        b6="$(printf %02x $(( 0x$b6 | 0x04 )))"
    fi
    if [ "$enablescanpassword" == "true" ]; then
        b6="$(printf %02x $(( 0x$b6 | 0x08 )))"
    fi
    if [ "$enablejtagpassword" == "true" ]; then
        b6="$(printf %02x $(( 0x$b6 | 0x10 )))"
    fi

    b7="00"
    if [ "$enableaes" == "true" ]; then
        b7="$(printf %02x $(( 0x$b7 | 0x10 )))"
    fi

    b8="00"
    if [ "$enableantirollback" == "true" ]; then
        b8="$(printf %02x $(( 0x$b8 | 0x01 )))"
    fi
    if [ "$enableusbpassword" == "true" ]; then
        b8="$(printf %02x $(( 0x$b8 | 0x02 )))"
    fi

    if [ "$soc" == "txlx" ]; then
       b8="$(printf %02x $(( 0x$b8 | 0x30 )))"
    fi

    b9="00"
    if [ "$disablescanchain" == "true" ]; then
        b9="$(printf %02x $(( 0x$b9 | 0x04 )))"
    fi

    if [ ${#b4} -ne 2 ] || [ ${#b5} -ne 2 ] || [ ${#b6} -ne 2 ] || [ ${#b7} -ne 2 ] || [ ${#b8} -ne 2 ] || [ ${#b9} -ne 2 ]; then
        echo Internal Error
        exit 1
    fi

    echo 00 00 00 00 $b4 $b5 $b6 $b7 $b8 $b9 00 00 00 00 00 00 | xxd -r -p > $TMP/license.bin

    filesize=$(wc -c < $TMP/license.bin)
    if [ $filesize -ne 16 ]; then
        echo Internal Error 2
        exit 1
    fi

    dd if=/dev/zero of=$TMP/patt.bin count=512 bs=1 &> /dev/null

    dd if=$TMP/license.bin of=$TMP/patt.bin bs=1 seek=0 count=16 \
        conv=notrunc >& /dev/null

    if [ "$roothash" != "" ]; then
        dd if=$roothash of=$TMP/patt.bin bs=1 seek=48 count=32 \
            conv=notrunc >& /dev/null
    fi

    if [ "$aeskey" != "" ]; then
        dd if=$aeskey of=$TMP/patt.bin bs=1 seek=80 count=32 \
            conv=notrunc >& /dev/null
    fi

    if [ "$jtagusbpasswordhash" != "" ]; then
        dd if=$jtagusbpasswordhash of=$TMP/patt.bin bs=1 seek=128 count=32 \
            conv=notrunc >& /dev/null
    fi

    if [ "$scanpasswordhash" != "" ]; then
        dd if=$scanpasswordhash of=$TMP/patt.bin bs=1 seek=160 count=32 \
            conv=notrunc >& /dev/null
    fi

    if [ "$userefusefile" != "" ]; then
        dd if=$userefusefile of=$TMP/patt.bin bs=1 seek=320 count=192 \
            conv=notrunc >& /dev/null
    fi

    ${TOOL_PATH}/aml_encrypt_gxl --efsgen3 --input $TMP/patt.bin --output $output --debug enable

    # MARCO - START - Add pattern for uboot and TA usages
    mv $output $output.uboot

    if [ "$aeskey" != "" ]; then
        cp $aeskey $TMP/aeskey.ta
        chmod u+w $TMP/aeskey.ta
    elif [ "$aeskeyta" != "" ]; then
        cp $aeskeyta $TMP/aeskey.ta
        chmod u+w $TMP/aeskey.ta
    else
        dd if=/dev/zero of=$TMP/aeskey.ta bs=32 count=1 >& /dev/null
    fi

    ${TOOL_PATH}/wrap_provisioning_data.py $TMP/aeskey.ta $TMP/patt.bin $output.ta >& /dev/null
    dd if=/dev/zero of=$TMP/aeskey.ta bs=32 count=1 >& /dev/null
    ${TOOL_PATH}/wrap_provisioning_data.py $TMP/aeskey.ta $TMP/patt.bin $output.first.ta >& /dev/null
    # MARCO - STOP - Add pattern for uboot and TA usages
}

generate_efuse_pattern_axg() {
    # Additional args check
    if [ "$userefusefile" != "" ]; then
        check_file userefusefile "$userefusefile"
        local filesize=$(wc -c < ${userefusefile})
        if [ $filesize -ne 16 ]; then
            echo "User efuse file incorrect size $filesize != 16"
            exit 1
        fi
    fi
    if [ "$disablebootrecover" == "true" ]; then
        echo Boot recovery is not supported on $soc, option ignored
    fi

    # Create input pattern
    b5="00"
    if [ "$disablebootusb" == "true" ]; then
        b5="$(printf %02x $(( 0x$b5 | 0x01 )))"
    fi
    if [ "$disablebootspi" == "true" ]; then
        b5="$(printf %02x $(( 0x$b5 | 0x02 )))"
    fi
    if [ "$disablebootsdcard" == "true" ]; then
        b5="$(printf %02x $(( 0x$b5 | 0x04 )))"
    fi
    if [ "$disablebootnandemmc" == "true" ]; then
        b5="$(printf %02x $(( 0x$b5 | 0x08 )))"
    fi
    if [ "$disableprint" == "true" ]; then
        b5="$(printf %02x $(( 0x$b5 | 0x10 )))"
    fi
    if [ "$disablejtag" == "true" ]; then
        b5="$(printf %02x $(( 0x$b5 | 0x20 )))"
    fi
    if [ "$enablejtagpassword" == "true" ]; then
        b5="$(printf %02x $(( 0x$b5 | 0x40 )))"
    fi
    if [ "$enableusbpassword" == "true" ]; then
        b5="$(printf %02x $(( 0x$b5 | 0x80 )))"
    fi

    b6="00"
    if [ "$enablesb" == "true" ]; then
        b6="$(printf %02x $(( 0x$b6 | 0x41 )))"
    fi
    if [ "$enableaes" == "true" ]; then
        b6="$(printf %02x $(( 0x$b6 | 0x02 )))"
    fi
    if $revokersk0; then
        b6="$(printf %02x $(( 0x$b6 | 0x04 )))"
    fi
    if $revokersk1; then
        b6="$(printf %02x $(( 0x$b6 | 0x08 )))"
    fi
    if $revokersk2; then
        b6="$(printf %02x $(( 0x$b6 | 0x10 )))"
    fi
    if $revokersk3; then
        b6="$(printf %02x $(( 0x$b6 | 0x20 )))"
    fi
    if [ "$enableantirollback" == "true" ]; then
        b6="$(printf %02x $(( 0x$b6 | 0x80 )))"
    fi

    b7="00"
    if [ "$disablescanchain" == "true" ]; then
        b7="$(printf %02x $(( 0x$b7 | 0x01 )))"
    fi
    if [ "$enablescanpassword" == "true" ]; then
        b7="$(printf %02x $(( 0x$b7 | 0x02 )))"
    fi

    if [ ${#b5} -ne 2 ] || [ ${#b6} -ne 2 ] || [ ${#b7} -ne 2 ]; then
        echo Internal Error
        exit 1
    fi

    echo 00 00 00 00 00 $b5 $b6 $b7 00 00 00 00 00 00 00 00 | xxd -r -p > $TMP/license.bin
    filesize=$(wc -c < $TMP/license.bin)
    if [ $filesize -ne 16 ]; then
        echo Internal Error 2
        exit 1
    fi

    dd if=/dev/zero of=$TMP/patt.bin count=512 bs=1 &> /dev/null

    dd if=$TMP/license.bin of=$TMP/patt.bin bs=1 seek=176 count=16 \
        conv=notrunc >& /dev/null

    if [ "$roothash" != "" ]; then
        dd if=$roothash of=$TMP/patt.bin bs=1 seek=32 count=32 \
            conv=notrunc >& /dev/null
    fi

    if [ "$aeskey" != "" ]; then
        dd if=$aeskey of=$TMP/patt.bin bs=1 seek=64 count=32 \
            conv=notrunc >& /dev/null
    fi

    if [ "$jtagusbpasswordhash" != "" ]; then
        dd if=$jtagusbpasswordhash of=$TMP/patt.bin bs=1 seek=96 count=32 \
            conv=notrunc >& /dev/null
    fi

    if [ "$scanpasswordhash" != "" ]; then
        dd if=$scanpasswordhash of=$TMP/patt.bin bs=1 seek=128 count=32 \
            conv=notrunc >& /dev/null
    fi

    if [ "$userefusefile" != "" ]; then
        dd if=$userefusefile of=$TMP/patt.bin bs=1 seek=240 count=16 \
            conv=notrunc >& /dev/null
    fi

    ${TOOL_PATH}/aml_encrypt_axg --efsproc --input $TMP/patt.bin --output $output

    # MARCO - START - Add pattern for uboot and TA usages
    dd if=$TMP/patt.bin of=$output.debug count=1 bs=256 &> /dev/null
    mv $output $output.uboot

    if [ "$aeskey" != "" ]; then
        cp $aeskey $TMP/aeskey.ta
        chmod u+w $TMP/aeskey.ta
    elif [ "$aeskeyta" != "" ]; then
        cp $aeskeyta $TMP/aeskey.ta
        chmod u+w $TMP/aeskey.ta
    else
        dd if=/dev/zero of=$TMP/aeskey.ta bs=32 count=1 >& /dev/null
    fi
    ${TOOL_PATH}/wrap_provisioning_data.py $TMP/aeskey.ta $output.debug $output.ta >& /dev/null
    dd if=/dev/zero of=$TMP/aeskey.ta bs=32 count=1 >& /dev/null
    ${TOOL_PATH}/wrap_provisioning_data.py $TMP/aeskey.ta $output.debug $output.first.ta >& /dev/null
    # MARCO - STOP - Add pattern for uboot and TA usages
}

parse_main() {
    case "$@" in
	*-o*)
            generate_efuse_pattern "$@"
	    ;;
	*)
            usage
            ;;
    esac
}

cleanup() {
    # MARCO - START - Fix cleanup
    rm -f $TMP/patt.bin
    rm -f $TMP/license.bin
    rm -f $TMP/aeskey.ta
    if [ -d $TMP ]; then
      rmdir $TMP
    fi
    return
    # MARCO - STOP  - Fix cleanup
}

trap cleanup EXIT

if [ ! -d "$TMP" ]; then mkdir "$TMP" ; fi
parse_main "$@"
cleanup
