#!/bin/bash
# script for generating signed swupdate file

set -e

usage() {
	echo "Usage:"
	echo "  $0 <path_to_images> [MACHINE] <DATETIME> <MODE>"
	echo "  <MODE>: can be 'fit', 'uboot', 'aml', 'swu' or 'all' to specify the parts to sign"
}

source "$(dirname $(readlink -f "$0"))/sign_swupdate_common.sh"

sign_fit() {
	insert_various_items_into_swu_rootfs

	echo "== Creating FIT images..."

	create_dtb_file_pubkey

	echo "==== Signing swupdate kernel/dts/optee images..."
	LD_LIBRARY_PATH="${YOCTO_LIBRARY_PATH}" ${MKIMAGE} -D "${DTC_OPS}" -f "${DEPLOY_DIR_IMAGE}/fitImage-its-swu-${CONFIG_MACHINE}.its" -K "${DTB_FILE_PUBKEY}.dtb" -k "${DEPLOY_DIR_IMAGE}/boot_keys/" -r "${DEPLOY_DIR_IMAGE}/fitImage-swu-${CONFIG_MACHINE}-${DATETIME}-signed.itb"
	ln -sf "fitImage-swu-${CONFIG_MACHINE}-${DATETIME}-signed.itb" "${DEPLOY_DIR_IMAGE}/fitImage-swu-${CONFIG_MACHINE}-signed"

	echo "==== Signing application kernel/dts/optee images..."
	LD_LIBRARY_PATH="${YOCTO_LIBRARY_PATH}" ${MKIMAGE} -D "${DTC_OPS}" -f "${DEPLOY_DIR_IMAGE}/fitImage-its-${CONFIG_MACHINE}.its" -K "${DTB_FILE_PUBKEY}.dtb" -k "${DEPLOY_DIR_IMAGE}/boot_keys/" -r "${DEPLOY_DIR_IMAGE}/fitImage-${CONFIG_MACHINE}-${DATETIME}-signed.itb"
	ln -sf "fitImage-${CONFIG_MACHINE}-${DATETIME}-signed.itb" "${DEPLOY_DIR_IMAGE}/fitImage-${CONFIG_MACHINE}-signed"

	echo "== Creating FIT images complete!"
}

sign_uboot() {
	echo "== Creating U-Boot image..."

	# insert TA minimum required version into bl32.img
	ANTI_ROLLBACK_VERSION_FILE="${TMP_DIR}/arbt.txt"
	IMAGE_SECURITY_VERSION=$(cat ${DEPLOY_DIR_IMAGE}/version.part.security | cut -d'.' -f4)
	# on ARCAM we want to neuter this mechanism for the time being, so the IMAGE_SECURITY_VERSION is set to 1
	IMAGE_SECURITY_VERSION=1
	[[ -z "$IMAGE_SECURITY_VERSION" ]] && { echo "IMAGE_SECURITY_VERSION is empty" ; exit 1; }
	for N in ${DEPLOY_DIR_IMAGE}/ta/*.stripped.elf; do
		[ -e "$N" ] || continue
		BN=`basename "$N" .stripped.elf`
		echo "==== $BN.."
		echo "${BN}:0.0.${IMAGE_SECURITY_VERSION}" >> "${ANTI_ROLLBACK_VERSION_FILE}"
	done

	if [ -s $ANTI_ROLLBACK_VERSION_FILE ]; then
		"${DEPLOY_DIR_IMAGE}/signing_scripts/pack_arbt.py" --in "${DEPLOY_DIR_IMAGE}/board/bl32.img" --out="${TMP_DIR}/bl32_with_TA_vers.img" --arb_table "${ANTI_ROLLBACK_VERSION_FILE}"
	else
		# no TAs found, nothing to insert
		cp "${DEPLOY_DIR_IMAGE}/board/bl32.img" "${TMP_DIR}/bl32_with_TA_vers.img"
	fi

	# generate zeros aes key, this will disable TA encryption
	dd if="/dev/zero" of="${TMP_DIR}/aes_key.bin" count=1 bs=32

	"${DEPLOY_DIR_IMAGE}/signing_scripts/pack_kpub.py" --rsk="${DEPLOY_DIR_IMAGE}/boot_keys/optee-ta.pubkey" --rek="${TMP_DIR}/aes_key.bin" \
		--in="${TMP_DIR}/bl32_with_TA_vers.img" --out="${TMP_DIR}/bl32_with_keys.img"

	create_dtb_file_pubkey

	cat "${DEPLOY_DIR_IMAGE}/board/bl33.bin" "${DTB_FILE_PUBKEY}.dtb" > "${TMP_DIR}/bl33_with_keys.bin"

	echo "=== Creating U-Boot with Optee keys..."

	install -d "${DEPLOY_DIR_IMAGE}/aml_files"

	AML_ENCRYPT_AXG="${DEPLOY_DIR_IMAGE}/signing_scripts/tools/aml_encrypt_axg"

	${AML_ENCRYPT_AXG} --bl2sig --input "${DEPLOY_DIR_IMAGE}/board/bl2.bin" --output "${TMP_DIR}/bl2.bin.axg"
	${AML_ENCRYPT_AXG} --bl3sig --input "${DEPLOY_DIR_IMAGE}/board/bl30.bin" --output "${TMP_DIR}/bl30.bin.axg" --level v3 --type bl30
	${AML_ENCRYPT_AXG} --bl3sig --input "${DEPLOY_DIR_IMAGE}/board/bl31.img" --output "${TMP_DIR}/bl31.img.axg" --level v3 --type bl31
	${AML_ENCRYPT_AXG} --bl3sig --input "${TMP_DIR}/bl32_with_keys.img" --output  "${TMP_DIR}/bl32_with_keys.img.axg" --level v3 --type bl32
	${AML_ENCRYPT_AXG} --bl3sig --input "${TMP_DIR}/bl33_with_keys.bin" --compress lz4 --output  "${TMP_DIR}/bl33_with_keys.bin.axg" --level v3 --type bl33

	${AML_ENCRYPT_AXG} --bootmk --output ${TMP_DIR}/u-boot-${CONFIG_MACHINE}-keys.bin \
		--bl2 "${TMP_DIR}/bl2.bin.axg" --bl30 "${TMP_DIR}/bl30.bin.axg" \
		--bl31 "${TMP_DIR}/bl31.img.axg" --bl32 "${TMP_DIR}/bl32_with_keys.img.axg" --bl33 "${TMP_DIR}/bl33_with_keys.bin.axg" --level v3

	cp "${TMP_DIR}/u-boot-${CONFIG_MACHINE}-keys.bin" "${DEPLOY_DIR_IMAGE}/u-boot-${CONFIG_MACHINE}-keys.bin"
	ln -sf "${DEPLOY_DIR_IMAGE}/u-boot-${CONFIG_MACHINE}-keys.bin" "${DEPLOY_DIR_IMAGE}/aml_files/u-boot.bin"

	dd if="${DEPLOY_DIR_IMAGE}/u-boot-${CONFIG_MACHINE}-keys.bin" of="${DEPLOY_DIR_IMAGE}/aml_files/u-boot.bin.usb.bl2" bs=49152 count=1 &> /dev/null
	dd if="${DEPLOY_DIR_IMAGE}/u-boot-${CONFIG_MACHINE}-keys.bin" of="${DEPLOY_DIR_IMAGE}/aml_files/u-boot.bin.usb.tpl" bs=49152 skip=1 &> /dev/null

	# sd.bin is used only by aml_upgrade_package.img
	dd if="/dev/zero" of="${TMP_DIR}/mbr.bin" bs=512 count=1 &> /dev/null
	cat "${TMP_DIR}/mbr.bin" "${DEPLOY_DIR_IMAGE}/u-boot-${CONFIG_MACHINE}-keys.bin" > "${DEPLOY_DIR_IMAGE}/aml_files/u-boot.bin.sd.bin"

	echo "=== Creating signed U-Boot with Optee keys..."

	ROOT_KEY_INDEX=0
	# hash version for axg or txlx
	KEY_HASH_VER=2

	# fip-key and bl2-key has to be same
	BL2_AND_FIP_KEY="bl2.pem"

	# signing script expects the security number to be specified in hex
	IMAGE_SECURITY_VERSION=$(printf "0x%x" "$(cat ${DEPLOY_DIR_IMAGE}/version.part.security | cut -d'.' -f4)")

	echo "==== Image security version: ${IMAGE_SECURITY_VERSION}"

	BL2_ARB_CVN="${IMAGE_SECURITY_VERSION}"
	BL30_ARB_CVN="${IMAGE_SECURITY_VERSION}"
	BL31_ARB_CVN="${IMAGE_SECURITY_VERSION}"
	BL32_ARB_CVN="${IMAGE_SECURITY_VERSION}"
	BL33_ARB_CVN="${IMAGE_SECURITY_VERSION}"
	FIP_ARB_CVN="${IMAGE_SECURITY_VERSION}"

	"${DEPLOY_DIR_IMAGE}/signing_scripts/sign-boot.sh" --create-signed-bl \
		--bl2            ${DEPLOY_DIR_IMAGE}/board/bl2.bin \
		--root-key-idx   ${ROOT_KEY_INDEX} \
		--root-key       ${DEPLOY_DIR_IMAGE}/boot_keys/root-$ROOT_KEY_INDEX.pem \
		--root-key-0     ${DEPLOY_DIR_IMAGE}/boot_keys/root-0.pem \
		--root-key-1     ${DEPLOY_DIR_IMAGE}/boot_keys/root-1.pem \
		--root-key-2     ${DEPLOY_DIR_IMAGE}/boot_keys/root-2.pem \
		--root-key-3     ${DEPLOY_DIR_IMAGE}/boot_keys/root-3.pem \
		--key-hash-ver   ${KEY_HASH_VER} \
		--bl2-key        ${DEPLOY_DIR_IMAGE}/boot_keys/$BL2_AND_FIP_KEY \
		--bl2-key-0      ${DEPLOY_DIR_IMAGE}/boot_keys/$BL2_AND_FIP_KEY \
		--bl2-key-1      ${DEPLOY_DIR_IMAGE}/boot_keys/$BL2_AND_FIP_KEY \
		--bl2-key-2      ${DEPLOY_DIR_IMAGE}/boot_keys/$BL2_AND_FIP_KEY \
		--bl2-key-3      ${DEPLOY_DIR_IMAGE}/boot_keys/$BL2_AND_FIP_KEY \
		--bl30           ${DEPLOY_DIR_IMAGE}/board/bl30.bin \
		--bl30-key       ${DEPLOY_DIR_IMAGE}/boot_keys/bl3x.pem \
		--bl31           ${DEPLOY_DIR_IMAGE}/board/bl31.img \
		--bl31-key       ${DEPLOY_DIR_IMAGE}/boot_keys/bl3x.pem \
		--bl32           ${TMP_DIR}/bl32_with_keys.img \
		--bl32-key       ${DEPLOY_DIR_IMAGE}/boot_keys/bl3x.pem \
		--bl33           ${TMP_DIR}/bl33_with_keys.bin \
		--bl33-key       ${DEPLOY_DIR_IMAGE}/boot_keys/bl3x.pem \
		--fip-key        ${DEPLOY_DIR_IMAGE}/boot_keys/$BL2_AND_FIP_KEY \
		--bl2-arb-cvn    ${BL2_ARB_CVN} \
		--bl30-arb-cvn   ${BL30_ARB_CVN} \
		--bl31-arb-cvn   ${BL31_ARB_CVN} \
		--bl32-arb-cvn   ${BL32_ARB_CVN} \
		--bl33-arb-cvn   ${BL33_ARB_CVN} \
		--fip-arb-cvn    ${FIP_ARB_CVN} \
		-e none \
		-o ${DEPLOY_DIR_IMAGE}/u-boot-${CONFIG_MACHINE}-keys-signed.bin

	"${DEPLOY_DIR_IMAGE}/signing_scripts/sign-boot.sh" --create-root-hash \
		--root-key-0 ${DEPLOY_DIR_IMAGE}/boot_keys/root-0.pem \
		--root-key-1 ${DEPLOY_DIR_IMAGE}/boot_keys/root-1.pem \
		--root-key-2 ${DEPLOY_DIR_IMAGE}/boot_keys/root-2.pem \
		--root-key-3 ${DEPLOY_DIR_IMAGE}/boot_keys/root-3.pem \
		--key-hash-ver $KEY_HASH_VER         \
		-o "${TMP_DIR}/rootkeys-hash.bin"

	"${DEPLOY_DIR_IMAGE}/signing_scripts/efuse-tool.sh" --generate-efuse-pattern \
		--soc                    axg \
		--root-hash              "${TMP_DIR}/rootkeys-hash.bin" \
		--enable-sb              true \
		--enable-aes             false \
		--enable-jtag-password   true \
		--enable-usb-password    true \
		--enable-scan-password   true \
		--enable-anti-rollback   false \
		--disable-boot-usb       false \
		--disable-boot-spi       true \
		--disable-boot-sdcard    true \
		--disable-boot-nand-emmc false \
		--disable-boot-recover   true \
		--disable-scan-chain     true \
		--disable-print          false \
		--disable-jtag           true \
		--revoke-rsk-0           false \
		--revoke-rsk-1           false \
		--revoke-rsk-2           false \
		--revoke-rsk-3           false \
		--jtag-usb-password-hash "${DEPLOY_DIR_IMAGE}/boot_keys/pwhash.bin" \
		--scan-password-hash     "${DEPLOY_DIR_IMAGE}/boot_keys/pwhash.bin" \
		-o ${TMP_DIR}/u-boot-${CONFIG_MACHINE}-efuse.bin

	install ${TMP_DIR}/u-boot-${CONFIG_MACHINE}-efuse.bin.uboot "${DEPLOY_DIR_IMAGE}/u-boot-${CONFIG_MACHINE}-efuse.bin.uboot"
	LD_LIBRARY_PATH="${YOCTO_LIBRARY_PATH}" "${DEPLOY_DIR_IMAGE}/signing_scripts/provision_keywrapper" -i ${TMP_DIR}/u-boot-${CONFIG_MACHINE}-efuse.bin.first.ta
	install ${TMP_DIR}/u-boot-${CONFIG_MACHINE}-efuse.bin.first.ta.enc "${DEPLOY_DIR_IMAGE}/u-boot-${CONFIG_MACHINE}-efuse.bin.first.ta.enc"

	ln -sf "${DEPLOY_DIR_IMAGE}/u-boot-${CONFIG_MACHINE}-keys-signed.bin" "${DEPLOY_DIR_IMAGE}/aml_files/u-boot.bin.signed"

	dd if="${DEPLOY_DIR_IMAGE}/u-boot-${CONFIG_MACHINE}-keys-signed.bin" of="${DEPLOY_DIR_IMAGE}/aml_files/u-boot.bin.usb.bl2.signed" bs=49152 count=1 &> /dev/null
	dd if="${DEPLOY_DIR_IMAGE}/u-boot-${CONFIG_MACHINE}-keys-signed.bin" of="${DEPLOY_DIR_IMAGE}/aml_files/u-boot.bin.usb.tpl.signed" bs=49152 skip=1 &> /dev/null

	# sd.bin is used only by aml_upgrade_package.img
	dd if="/dev/zero" of="${TMP_DIR}/mbr.bin" bs=512 count=1 &> /dev/null
	cat "${TMP_DIR}/mbr.bin" "${DEPLOY_DIR_IMAGE}/u-boot-${CONFIG_MACHINE}-keys-signed.bin" > "${DEPLOY_DIR_IMAGE}/aml_files/u-boot.bin.sd.bin.signed"

	# pack signed and unsigned u-boot to one file
	mkdir "${TMP_DIR}/uboot"
	cp "${DEPLOY_DIR_IMAGE}/u-boot-${CONFIG_MACHINE}-keys.bin" "${TMP_DIR}/uboot/u-boot.bin"
	cp "${DEPLOY_DIR_IMAGE}/u-boot-${CONFIG_MACHINE}-keys-signed.bin" "${TMP_DIR}/uboot/u-boot-signed.bin"

	tar czf "${DEPLOY_DIR_IMAGE}/u-boot.tar.gz" -C "${TMP_DIR}" "uboot/"

	echo "== Creating U-Boot image complete!"
}

sign_ta() {
	echo "== Signing all TAs..."
	mkdir -p "${TMP_DIR}/ta"

	# generate random rsa private key
	openssl genpkey -algorithm RSA -out "${TMP_DIR}/ta/rsa.key" -pkeyopt rsa_keygen_bits:2048
	#dd if="/dev/urandom" of="${TMP_DIR}/ta/ta_aes.bin" count=1 bs=32
	#dd if="/dev/urandom" of="${TMP_DIR}/ta/ta_iv.bin" count=1 bs=16

	IMAGE_SECURITY_VERSION=$(cat ${DEPLOY_DIR_IMAGE}/version.part.security | cut -d'.' -f4)
	# on ARCAM we want to neuter this mechanism for the time being, so the IMAGE_SECURITY_VERSION is set to 1
	IMAGE_SECURITY_VERSION=1
	[[ -z "$IMAGE_SECURITY_VERSION" ]] && { echo "IMAGE_SECURITY_VERSION is empty" ; exit 1; }

	for N in ${DEPLOY_DIR_IMAGE}/ta/*.stripped.elf; do
		[ -e "$N" ] || continue
		BN=`basename "$N" .stripped.elf`
		echo "==== $BN.."

		${DEPLOY_DIR_IMAGE}/signing_scripts/gen_cert_key.py \
			--root_rsa_key="${DEPLOY_DIR_IMAGE}/boot_keys/optee-ta.key" \
			--uuid="$BN" \
			--ta_rsa_key="${TMP_DIR}/ta/rsa.key" \
			--ta_rsa_key_sig="${TMP_DIR}/ta/sig"
			#--root_aes_key="${TMP_DIR}/ta/root_aes.bin"
			#--ta_aes_key="${TMP_DIR}/ta/ta_aes.bin" \
			#--ta_aes_iv="${TMP_DIR}/ta/ta_iv.bin" \
			#--ta_aes_key_iv_enc="${TMP_DIR}/ta/enc"

		${DEPLOY_DIR_IMAGE}/signing_scripts/nosign.py --in="$N" --out="${TMP_DIR}/ta/$BN.ta.nosign"

		${DEPLOY_DIR_IMAGE}/signing_scripts/sign_ta.py --ta_rsa_key="${TMP_DIR}/ta/rsa.key" \
			--ta_rsa_key_sig="${TMP_DIR}/ta/sig" \
			--in="${TMP_DIR}/ta/$BN.ta.nosign" \
			--out="${TMP_DIR}/ta/$BN.ta" \
			--ta_cvn="0.0.${IMAGE_SECURITY_VERSION}"
			#--ta_aes_key="${TMP_DIR}/ta/ta_aes.bin" \
			#--ta_aes_iv="${TMP_DIR}/ta/ta_iv.bin" \
			#--ta_aes_key_iv_enc="${TMP_DIR}/ta/enc"
	done
}

create_aml_file() {
	echo "== Creating aml upgrade package"
	ln -sf "${DEPLOY_DIR_IMAGE}/fitImage-swu-${CONFIG_MACHINE}-signed" "${DEPLOY_DIR_IMAGE}/aml_files/fitImage-swu-signed"
	ln -sf "${DEPLOY_DIR_IMAGE}/fitImage-${CONFIG_MACHINE}-signed" "${DEPLOY_DIR_IMAGE}/aml_files/fitImage-signed"
	ln -sf "${DEPLOY_DIR_IMAGE}/board/aml_sdc_burn.ini" "${DEPLOY_DIR_IMAGE}/aml_files/aml_sdc_burn.ini"
	ln -sf "${DEPLOY_DIR_IMAGE}/board/platform.conf" "${DEPLOY_DIR_IMAGE}/aml_files/platform.conf"
	ln -sf "${DEPLOY_DIR_IMAGE}/board/swuenv.img" "${DEPLOY_DIR_IMAGE}/aml_files/swuenv.img"
	ln -sf "${DEPLOY_DIR_IMAGE}/yocto-nsdk-ip-image-swu-${CONFIG_MACHINE}.ubifs" "${DEPLOY_DIR_IMAGE}/aml_files/yocto-nsdk-ip-image-swu.ubifs"
	MKUBIFS_ARGS=`cat "${DEPLOY_DIR_IMAGE}/signing_scripts/params.mkubifs"`
	"${DEPLOY_DIR_IMAGE}"/signing_scripts/mkfs.ubifs -o "${DEPLOY_DIR_IMAGE}/aml_files/empty.ubifs" ${MKUBIFS_ARGS}
	sed -i "
		s|^image=.*rootfs\.ubifs\$|image=${DEPLOY_DIR_IMAGE}/aml_files/yocto-nsdk-ip-image-swu.ubifs|;
		s|^image=.*empty\.ubifs\$|image=${DEPLOY_DIR_IMAGE}/aml_files/empty.ubifs|;
	" "${DEPLOY_DIR_IMAGE}"/board/ubinize.cfg
	UBINIZE_ARGS='-m 2048 -p 131072'
	ubinize -o "${DEPLOY_DIR_IMAGE}/aml_files/data.ubi" ${UBINIZE_ARGS} "${DEPLOY_DIR_IMAGE}"/board/ubinize.cfg

	# build axg dts to satify aml flash tool and u-boot
	dtc -p 0x1000 "${DEPLOY_DIR_IMAGE}/board/axg.dts" -O dtb -o "${DEPLOY_DIR_IMAGE}/aml_files/axg.dtb"

	# Windows only! If keys.conf contains "amlogic_set" keyword it looks for amlogic_set file containing e-fuses to be burn
	echo "amlogic_set" > "${DEPLOY_DIR_IMAGE}/aml_files/keys.conf"
	"${DEPLOY_DIR_IMAGE}/signing_scripts/aml_image_v2_packer_new" -r "${DEPLOY_DIR_IMAGE}/board/aml_upgrade_package_signed.conf" "${DEPLOY_DIR_IMAGE}/aml_files/" "${DEPLOY_DIR_IMAGE}/aml_upgrade_package_signed-${CONFIG_MACHINE}-${DATETIME}.img"
	ln -sf "aml_upgrade_package_signed-${CONFIG_MACHINE}-${DATETIME}.img" "${DEPLOY_DIR_IMAGE}/aml_upgrade_package_signed.img"

	echo "" > "${DEPLOY_DIR_IMAGE}/aml_files/keys.conf"
	"${DEPLOY_DIR_IMAGE}/signing_scripts/aml_image_v2_packer_new" -r "${DEPLOY_DIR_IMAGE}/board/aml_upgrade_package.conf" "${DEPLOY_DIR_IMAGE}/aml_files/" "${DEPLOY_DIR_IMAGE}/aml_upgrade_package-${CONFIG_MACHINE}-${DATETIME}.img"
	ln -sf "aml_upgrade_package-${CONFIG_MACHINE}-${DATETIME}.img" "${DEPLOY_DIR_IMAGE}/aml_upgrade_package.img"
	rm "${DEPLOY_DIR_IMAGE}/aml_files/keys.conf"
	echo "== Creating aml upgrade package complete!"
}

insert_platform_specific_items_into_rootfs() {
	local TMP_ROOTFS="$1"

	# make sure the destination dir exists and copy eFuse configuration to rootfs
	mkdir -p "${TMP_ROOTFS}/usr/share/factory-tool/files/"
	cp "${DEPLOY_DIR_IMAGE}/u-boot-${CONFIG_MACHINE}-efuse.bin.first.ta.enc" "${TMP_ROOTFS}/usr/share/factory-tool/files/efuses.bin"
	cp "${DEPLOY_DIR_IMAGE}/u-boot-${CONFIG_MACHINE}-keys-signed.bin" "${TMP_ROOTFS}/usr/share/factory-tool/files/u-boot-signed.bin"
}

MKIMAGE="${DEPLOY_DIR_IMAGE}/signing_scripts/mkimage-a113-${CONFIG_MACHINE}"
DTC_OPS="-I dts -O dtb -p 2000"

case "$MODE" in
	"fit")
		sign_fit
		;;

	"uboot")
		sign_uboot
		;;

	"swu")
		create_configured_swupdate_files
		;;

	"aml")
		create_aml_file
		;;

	"ta")
		sign_ta
		insert_various_items_into_rootfs
		insert_various_items_into_swu_rootfs
		;;

	"insert_keys_swu")
		insert_various_items_into_swu_rootfs
		;;

	"extract_rootfs")
		setup_imagename
		extract_rootfs_tarball "${UNPACKED_ROOTFS_PATH}"
		;;

	"all")
		sign_ta
		sign_fit
		sign_uboot

		insert_various_items_into_rootfs

		create_configured_swupdate_files

		create_aml_file
		;;

	*)
		echo "Error: unknown mode selected!"
		usage
		exit 1

esac
