#!/bin/bash
# common parts of script for generating signed swupdate file

set -e

source "$(dirname $(readlink -f "$0"))/sign_swupdate_common.config"

# use parallel gzip if available
[ -x "$(command -v pigz)" ] && GZIP=pigz || GZIP=gzip


create_dtb_file_pubkey() {
	# we need to create a minimalistic device-tree file where mkimage will store the
	# public key in the right format (to be appended to u-boot binary then)
	DTB_FILE_PUBKEY="${DEPLOY_DIR_IMAGE}/fitImage-pubkey-${CONFIG_MACHINE}-${DATETIME}"

	if [ -e "${DTB_FILE_PUBKEY}.dtb" ]; then
		echo "==== DTB for the public key already exists, reusing..."
	else
		echo "==== Creating DTB for the public key..."
		cat << EOF > "${DTB_FILE_PUBKEY}".dts
/dts-v1/;
/ {};
EOF

		dtc -p 0x1000 "${DTB_FILE_PUBKEY}.dts" -O dtb -o "${DTB_FILE_PUBKEY}.dtb"
	fi
}

inject_wrap_key_hmac_into_swupdate() {
	local TMP_ROOTFS="$1"
	if [ ${#WRAP_KEY} -eq 64 ] && [ ${#WRAP_HMAC} -eq 32 ] && [ -f "${TMP_ROOTFS}/usr/bin/swupdate" ]; then
		OFFSET=$(grep --byte-offset --only-matching --text -E 'a{16}b{32}' "${TMP_ROOTFS}/usr/bin/swupdate" | sed 's/:.*//')

		dd if="${TMP_ROOTFS}/usr/bin/swupdate" count=1 bs=${OFFSET} > "${TMP_ROOTFS}/usr/bin/swupdate_tmp"
		echo -n "${WRAP_HMAC}${WRAP_KEY}" | xxd -r -p >> "${TMP_ROOTFS}/usr/bin/swupdate_tmp"
		dd if="${TMP_ROOTFS}/usr/bin/swupdate" bs=$((${OFFSET} + 48))  skip=1 >> "${TMP_ROOTFS}/usr/bin/swupdate_tmp"
		mv "${TMP_ROOTFS}/usr/bin/swupdate_tmp" "${TMP_ROOTFS}/usr/bin/swupdate"
		chmod +x "${TMP_ROOTFS}/usr/bin/swupdate"
	fi
}

insert_trusted_apps_into_rootfs() {
	# Inserts all trusted applications into rootfs, which is specified by $1

	local ROOTFS="$1"
	if [ -z "${ROOTFS}" ]; then
		echo "Error: no rootfs directory specified to insert the trusted applications into!"
		exit 1
	fi

	local TALIST=""
	for N in ${TMP_DIR}/ta/*.ta; do
		[ -e "$N" ] || continue
		BN=`basename "$N"`
		TALIST="$TALIST $BN"
	done

	if [ -n "$TALIST" ]; then
		echo "==== Inserting trusted applications into the rootfs (${ROOTFS})..."

		# make sure the destination dir exists
		mkdir -p "${ROOTFS}/lib/optee_armtz"

		for BN in $TALIST; do
			echo "==== Installing $BN..."
			cp "${TMP_DIR}/ta/$BN" "${ROOTFS}/lib/optee_armtz/$BN"
		done
	else
		echo "==== No trusted applications to insert into the rootfs (${ROOTFS})"
	fi
}

insert_various_items_into_swu_rootfs() {
	echo "== Packing image verification key into SWU image..."

	local TMP_SWUPDATE_ROOTFS="${TMP_DIR}/swu_rootfs"
	mkdir -p "${TMP_SWUPDATE_ROOTFS}"

	echo "==== Extracting swupdate rootfs to a temporary directory..."
	zcat "${DEPLOY_DIR_IMAGE}/swupdate-image-${CONFIG_MACHINE}.cpio.gz" | (cd "${TMP_SWUPDATE_ROOTFS}" && cpio -id)

	echo "==== Inserting verification key into the swupdate rootfs..."
	openssl rsa -in "${DEPLOY_DIR_IMAGE}/boot_keys/verified-boot.key" -pubout -outform PEM -out "${TMP_SWUPDATE_ROOTFS}/home/root/verified-boot.pem"

	inject_wrap_key_hmac_into_swupdate "${TMP_SWUPDATE_ROOTFS}"

	insert_trusted_apps_into_rootfs "${TMP_SWUPDATE_ROOTFS}"

	echo "==== Packing the swupdate rootfs back together..."
	(cd "${TMP_SWUPDATE_ROOTFS}" && find . | cpio -R 'root:' -o -H newc) | ${GZIP} -f -9 > "${DEPLOY_DIR_IMAGE}/swupdate-image-${CONFIG_MACHINE}-${DATETIME}-keys.rootfs.cpio.gz"
	ln -sf "swupdate-image-${CONFIG_MACHINE}-${DATETIME}-keys.rootfs.cpio.gz" "${DEPLOY_DIR_IMAGE}/swupdate-image-${CONFIG_MACHINE}-keys.cpio.gz"
}

setup_imagename()
{
	# Differentiate between ubifs and ext4
	if grep -q "\-${CONFIG_MACHINE}.ubifs" "${DEPLOY_DIR_IMAGE}/sw-description"; then
		ROOTFS_TYPE="ubifs"
	elif grep -q "\-${CONFIG_MACHINE}.ext4" "${DEPLOY_DIR_IMAGE}/sw-description"; then
		ROOTFS_TYPE="ext4"
	else
		echo "Error: Unsupported rootfs type!"
		exit 1
	fi
	#           grep for all lines containing valid filename entry                 | extract only the filename    | grep only for rootfs       | extract only imagename
	IMAGENAME=(`grep -o "filename[ ]*=[^;]*;" "${DEPLOY_DIR_IMAGE}/sw-description" | sed 's:.*"\(.*\)"[^"]*:\1:g' | grep "\-${CONFIG_MACHINE}.${ROOTFS_TYPE}$" | sed "s:-${CONFIG_MACHINE}.${ROOTFS_TYPE}\$::g"`)
}

extract_rootfs_tarball()
{
	local ROOTFS="$1"
	if [ -z "${ROOTFS}" ]; then
		echo "Error: no directory specified to extract rootfs to!"
		exit 1
	fi
	mkdir -p "${ROOTFS}"
	bzip2 -cd "${DEPLOY_DIR_IMAGE}/${IMAGENAME}-${CONFIG_MACHINE}.tar.bz2" | ${FAKEROOT} tar xf - -C "${ROOTFS}"
}

maybe_encrypt_default_settings() {
	# Performs default (read-only) settings encryption if it was requested.
	local ROOTFS="$1"

	if [ "${CONFIG_SETTINGS_ENCRYPTION}" != true ]; then
		return
	fi

	echo "==== Encrypting default settings"

	"${DEPLOY_DIR_IMAGE}/key_scripts/encrypt_settings.py" --key "${DEPLOY_DIR_IMAGE}/boot_keys/master_key.pubkey" --settingsDirectory "${ROOTFS}/settings-default"

	echo "==== Settings encryption completed"
}

insert_various_items_into_rootfs()
{
	echo "== Packing signed TAs and fit public key into rootfs image..."
	setup_imagename

	local TMP_ROOTFS
	if [ -n "${UNPACKED_ROOTFS_PATH}" -a -d "${UNPACKED_ROOTFS_PATH}" ]; then
		echo "==== Using pre-extracted rootfs tarball..."
		TMP_ROOTFS="${UNPACKED_ROOTFS_PATH}"
	else
		echo "==== Extracting rootfs tarball..."
		TMP_ROOTFS="${TMP_DIR}/rootfs-unpacked"
		extract_rootfs_tarball "${TMP_ROOTFS}"
	fi

	insert_trusted_apps_into_rootfs "${TMP_ROOTFS}"

	# make sure the destination dir exists
	mkdir -p "${TMP_ROOTFS}/home/root"
	echo "==== Inserting verification keys into rootfs..."
	cp "${DEPLOY_DIR_IMAGE}/fitImage-pubkey-${CONFIG_MACHINE}-${DATETIME}.dtb" "${TMP_ROOTFS}/home/root/verified-boot.dtb"
	openssl rsa -in "${DEPLOY_DIR_IMAGE}/boot_keys/verified-boot.key" -pubout -outform PEM -out "${TMP_ROOTFS}/home/root/verified-boot.pem"

	inject_wrap_key_hmac_into_swupdate "${TMP_ROOTFS}"

	insert_platform_specific_items_into_rootfs "${TMP_ROOTFS}"

	maybe_encrypt_default_settings "${TMP_ROOTFS}"

	if [ "$ROOTFS_TYPE" = "ubifs" ]; then
		OUT="${DEPLOY_DIR_IMAGE}/${IMAGENAME}-${CONFIG_MACHINE}-${DATETIME}.rootfs.ubifs"
		MKUBIFS_ARGS=`cat "${DEPLOY_DIR_IMAGE}/signing_scripts/params.mkubifs"`
		echo "==== Re-creating ubifs image..."
		${FAKEROOT} "${DEPLOY_DIR_IMAGE}/signing_scripts/mkfs.ubifs" ${MKUBIFS_ARGS} -r "${TMP_ROOTFS}" -o "$OUT"
		ln -sf "${IMAGENAME}-${CONFIG_MACHINE}-${DATETIME}.rootfs.ubifs" "${DEPLOY_DIR_IMAGE}/${IMAGENAME}-${CONFIG_MACHINE}.ubifs"
	else
		OUT="${DEPLOY_DIR_IMAGE}/${IMAGENAME}-${CONFIG_MACHINE}-${DATETIME}.rootfs.ext4"
		local IMG_OVERHEAD_FACTOR="13/10" # 1.3 the same in yocto
		local ACTUAL_ROTFS_SIZE=$(du -s ${TMP_ROOTFS} | awk '{print $1}')
		local ROOTFS_SIZE=$(($ACTUAL_ROTFS_SIZE * $IMG_OVERHEAD_FACTOR ))

		echo "==== Re-creating ext4 image..."
		dd if=/dev/zero of=${OUT} seek=${ROOTFS_SIZE} count=0 bs=1024
		extra_imagecmd="-i 4096" #from yocto
		${FAKEROOT} "mkfs.ext4" -F $extra_imagecmd ${OUT} -d "${TMP_ROOTFS}"
		# Error codes 0-3 indicate successfull operation of fsck (no errors or errors corrected
		fsck.ext4 -pvfD ${OUT} || [ $? -le 3 ]

		ln -sf "${IMAGENAME}-${CONFIG_MACHINE}-${DATETIME}.rootfs.ext4" "${DEPLOY_DIR_IMAGE}/${IMAGENAME}-${CONFIG_MACHINE}.ext4"
	fi

}

create_configured_swupdate_files() {
	# expecting:
	#	DEPLOY_DIR_IMAGE/sw-description
	#	DEPLOY_DIR_IMAGE/boot_keys/verified-boot.key
	# output:
	#	DEPLOY_DIR_IMAGE/image-${CONFIG_MACHINE}-${DATETIME}-signed.swu
	#	DEPLOY_DIR_IMAGE/image.swu (link)

	echo "== Creating swupdate${COMP} file..."

	ENCRYPTED_DIR="${TMP_DIR}/ENC"
	mkdir "${ENCRYPTED_DIR}"

	# Parse file names from sw-description
	DESC_FILES=(`grep -o "filename[ ]*=[^;]*;" "${DEPLOY_DIR_IMAGE}/sw-description" | sed 's:.*"\(.*\)"[^"]*:\1:g'`)

	echo "==== Copying files to a temporary directory..."
	for FILE in ${DESC_FILES[@]}; do
		FILE_PATH="${DEPLOY_DIR_IMAGE}/${FILE}"
		if [ "${FILE: -6}" = ".ubifs" -o "${FILE: -5}" = ".ext4" ]; then
			# NOTE: only use xz compression level "-6": this requires 9MB of ram for decompression
			#       (-9 requires 65MB which is too much for us)
			xz -T0 -6 -c "$FILE_PATH" > "${TMP_DIR}/${FILE}.xz"
			FILE_PATH="${TMP_DIR}/${FILE}.xz"
		fi

		cp "$FILE_PATH" "${TMP_DIR}/${FILE}"
		if [ "$CONFIG_USE_ENCRYPTION" == "true" ] && [ "${FILE: -3}" != ".sh" ]; then
			openssl enc -aes-256-cbc -in "$FILE_PATH" -out "${ENCRYPTED_DIR}/${FILE}" -K "${AES_KEY}" -iv "${AES_IV}" -nosalt
		else
			ln -sf "${TMP_DIR}/${FILE}" "${ENCRYPTED_DIR}/${FILE}"
		fi
	done

	# always create un-encrypted image
	ENCRYPTION=false ARTIFACTS_DIR="${TMP_DIR}" pack_swupdate_file

	# if encryption keys are available, also create encrypted image
	if [ "${CONFIG_USE_ENCRYPTION}" = true ]; then
		echo "== Creating swupdate-enc${COMP} file..."
		ENCRYPTION="${CONFIG_USE_ENCRYPTION}" ARTIFACTS_DIR="${ENCRYPTED_DIR}" pack_swupdate_file
	fi
}

pack_swupdate_file() {
	cp "${DEPLOY_DIR_IMAGE}/sw-description" "${ARTIFACTS_DIR}/sw-description"

	# Replace @filename with properties of the file (sha256, fileSize, sha256 of decrypted file, encryption enabled)
	echo "==== Calculating SHA-256 hashes for the sw-description file..."
	for FILE in ${DESC_FILES[@]}; do
		PROPERTIES=""
		REPLACE="sha256 = \"$(sha256sum ${ARTIFACTS_DIR}/$FILE | sed 's: .*::g')\";\n\
			filesize = \"$(stat -L -c%s ${DEPLOY_DIR_IMAGE}/$FILE)\";"
		if [ "${FILE: -6}" = ".ubifs" -o "${FILE: -5}" = ".ext4" ]; then
			REPLACE="$REPLACE\n\
			compressed = true;"
			PROPERTIES="\n\
			\tdecompressed-size = \"$(stat -L -c%s ${DEPLOY_DIR_IMAGE}/$FILE)\";"
		fi
		if [ "$ENCRYPTION" == "true" ] && [ "${FILE: -3}" != ".sh" ]; then
			REPLACE="$REPLACE\n\
			sha256dec = \"$(sha256sum ${DEPLOY_DIR_IMAGE}/$FILE | sed 's: .*::g')\";\n\
			encrypted = true;\
			"
			PROPERTIES="${PROPERTIES}\n\
			\tdecrypted-size = \"$(stat -L -c%s ${DEPLOY_DIR_IMAGE}/$FILE)\";"
		fi
		if [ "$PROPERTIES" != "" ]; then
			REPLACE="${REPLACE}\n\
			properties\: \{${PROPERTIES}\n\
			\}\n"
		fi
		sed -i "s:@$FILE:$REPLACE:g" "${ARTIFACTS_DIR}/sw-description"
	done

	# Update uboot and fit versions
	VERIFY_KEY_HASH=$(sha256sum "${DEPLOY_DIR_IMAGE}/boot_keys/verified-boot.key" | cut -d ' ' -f 1)
	sed -i "s:@VERKEY:x${VERIFY_KEY_HASH:0:7}:g" ${ARTIFACTS_DIR}/sw-description

	echo "==== Signing sw-description..."
	openssl dgst -sha256 -sign "${DEPLOY_DIR_IMAGE}/boot_keys/verified-boot.key" "${ARTIFACTS_DIR}/sw-description" > "${ARTIFACTS_DIR}/sw-description.sig"

	echo "==== Packing files into the final image..."
	[ "$ENCRYPTION" == "true" ] && POSTFIX="-enc.swu" || POSTFIX=".swu"

	# sw-descriptions have to come first
	(cd ${ARTIFACTS_DIR} && echo "sw-description sw-description.sig ${DESC_FILES[@]}" | tr ' ' '\n' | cpio -o -L -H crc) > "${DEPLOY_DIR_IMAGE}/image-${CONFIG_MACHINE}-${DATETIME}-signed${POSTFIX}"
	ln -sf "image-${CONFIG_MACHINE}-${DATETIME}-signed${POSTFIX}" "${DEPLOY_DIR_IMAGE}/image${POSTFIX}"

	echo "== Creating swupdate file complete!"
}


# beginning of the script

# global variables
DEPLOY_DIR_IMAGE="$1"
# make absolute path from it - we need to cd/pushd for mkimage run so make sure we use proper paths
# (realpath is not installed by default on ubuntu so just check for leading /)
[ "${DEPLOY_DIR_IMAGE:0:1}" == "/" ] || DEPLOY_DIR_IMAGE="`pwd`/$DEPLOY_DIR_IMAGE"

# Executables in the DEPLOY_DIR may be using uninative interpreter, which contains hardcoded path with the yocto tmp
# directory. This will cause error, when they are run at the customer or received via shared sstate cache from
# buildbot. To solve this, we change interpreter of all such executables to the default system interpreter.
if command -v patchelf &> /dev/null; then
	INTERPRETER="$(patchelf --print-interpreter `which patchelf`)"
	for FILE in $(find "$DEPLOY_DIR_IMAGE" -executable -type f | grep -v "\.\(sh\|py\)$"); do
		if patchelf --print-interpreter "$FILE" &> /dev/null; then
			if patchelf --print-interpreter "$FILE" | grep -q "uninative"; then
				patchelf --set-interpreter "$INTERPRETER" "$FILE"
			fi
		fi
	done
fi

if [ $# -eq 3 ]; then
	DATETIME="$2"
	MODE="$3"
elif [ $# -eq 4 ]; then
	if [ "$2" != "$CONFIG_MACHINE" ]; then
		echo "Error: specified MACHINE does not match the config ($2 != $CONFIG_MACHINE)!"
		exit 1
	fi
	DATETIME="$3"
	MODE="$4"
else
	echo "Error: invalid number of parameters!"
	usage
	exit 1
fi


# optional image encryption support
if [ "${CONFIG_USE_ENCRYPTION}" = true ]; then
	echo "== Encryption enabled"

	if [ ! -f "${DEPLOY_DIR_IMAGE}/swu_keys/enc_key" ]; then
		echo "Missing enc_key"
		exit 1
	fi

	if [ ! -f "${DEPLOY_DIR_IMAGE}/swu_keys/wrap_key" ]; then
		echo "Missing wrap_key"
		exit 1
	fi

	WRAP_KEY=$(grep "^key *= *" "${DEPLOY_DIR_IMAGE}/swu_keys/wrap_key" | sed 's/^key *= *//')
	WRAP_HMAC=$(grep "^hmac *= *" "${DEPLOY_DIR_IMAGE}/swu_keys/wrap_key" | sed 's/^hmac *= *//')
	if [ ${#WRAP_KEY} -ne 64 ] || [ ${#WRAP_HMAC} -ne 32 ]; then
		echo "Invalid wrap_key format"
		exit 1
	fi

	AES_KEY=$(grep "^key *= *" "${DEPLOY_DIR_IMAGE}/swu_keys/enc_key" | sed 's/^key *= *//')
	AES_IV=$(grep "^iv *= *" "${DEPLOY_DIR_IMAGE}/swu_keys/enc_key" | sed 's/^iv *= *//')
	if [ -z "${AES_KEY}" ] || [ -z "${AES_IV}" ]; then
		echo "Invalid aes key file format"
		exit 1
	fi
fi


# force paths for mkimage to local libs (ssl, crypto)
YOCTO_LIBRARY_PATH="${DEPLOY_DIR_IMAGE}/signing_scripts"


# create temp dir to be used during the image generation
TMP_DIR=`mktemp -d -t`
trap "rm -rf $TMP_DIR; exit" SIGHUP SIGINT SIGTERM EXIT

# setup fakeroot state-file so that re-creating the rootfs keeps correct permissions/owner information
if [ -z "${FAKEROOT_STATE_FILE}" ]; then
	FAKEROOT_STATE_FILE="${TMP_DIR}/_fakeroot_state"
fi
FAKEROOT_BINARY="/usr/bin/fakeroot"
FAKEROOT="${FAKEROOT_BINARY} -i ${FAKEROOT_STATE_FILE} -s ${FAKEROOT_STATE_FILE}"
