#! /bin/sh

# ALWAYS remove the USB upgrade flag on boot!
fw_setenv usb_update_req ""

##                                  ##
# AVATAR SOFTWARE UPGRADE DOWNLOADED #
##                                  ##

AVATAR_UPGRADE_COUNTER="/swupdateCounter"
AVATAR_UPGRADE_FAILED_FW="/swupdateFailedFw"
AVATAR_UPGRADE_DOWNLOADED_FLAG="/swupdateDownloaded"
AVATAR_UPGRADE_FILE_LOCATION="/swupdate"

AVATAR_UPGRADE_IN_PROGRESS_FLAG="/media/settings/settings/sentAvatarUpgradeInProgress"

# if the upgrade downloaded flag is set...
if [ -f $AVATAR_UPGRADE_DOWNLOADED_FLAG ]; then
    # remove the flag for future boots first
    rm -f $AVATAR_UPGRADE_DOWNLOADED_FLAG
    echo "AvatarUpgradeDownloaded flag file removed."

    # if the downloaded file exists, reboot to swupdate
    if [ -f $AVATAR_UPGRADE_FILE_LOCATION ]; then
        # set the upgrade in progress flag
        touch $AVATAR_UPGRADE_IN_PROGRESS_FLAG
        echo "AvatarUpgradeInProgress flag file set."

        # notify the MCU about entering the upgrade
        # State: 0x030B Avatar Upgrade Status
        # Encoding: uint8 Value: 0x05 (Avatar Upgrade In Progress)
        hostlink_cli -T 0 -C set_state 0x030B uint8 4

        # set the necessary environment flags and reboot
        fw_setenv update_url ubi://ubi0:nsdk-rootfs/swupdate
        fw_setenv update_url_cache ubi://ubi0:nsdk-rootfs/swupdate
        sfuflags -f update -v 1
        reboot
    else
        # otherwise, print warning (this _shouldn't_ ever happen)
        echo "WARNING: AvatarUpgradedDownloaded flag set, but $AVATAR_UPGRADE_FILE_LOCATION can't be found."
    fi
fi

##                                ##
# AVATAR SOFTWARE UPGRADE PROGRESS #
##                                ##

# retrieve build version strings; last boot and current
previousVersion=`cat /media/settings/settings/versionLastBooted  | jq -r .value.string_`
currentVersion=`cat /settings-default/version | jq -r .value.string_`

# if we had just performed a software upgrade, inform the MCU about it
if [ "x$previousVersion" != "x$currentVersion" ]; then
    echo "Successful upgrade from $previousVersion to $currentVersion; notifying MCU!"

    # State: 0x030B Avatar Upgrade Status
    # Encoding: uint8 Value: 0x05 (Avatar Upgrade Complete)
    hostlink_cli -T 0 -C set_state 0x030B uint8 5
fi

# if the flag file is set ...
if [ -f $AVATAR_UPGRADE_IN_PROGRESS_FLAG ]; then
    # ... but the version hasn't changed, our update must've failed,
    # so we need to inform the MCU about it
    if [ "x$previousVersion" == "x$currentVersion" ]; then
        echo "Failed to upgrade from $previousVersion; notifying MCU!"

        # State: 0x030B Avatar Upgrade Status
        # Encoding: uint8 Value: 0x06 (Avatar Upgrade Failure)
        hostlink_cli -T 0 -C set_state 0x030B uint8 6
    fi

    # ... clean the flag up
    rm -f $AVATAR_UPGRADE_IN_PROGRESS_FLAG
    echo "AvatarUpgradeInProgress flag file removed."
fi

##                            ##
# HOST SOFTWARE UPGRADE STATUS #
##                            ##

FIRMWARE_BASE_PATH="/lib/firmware"
VERSIONS_JSON=$FIRMWARE_BASE_PATH"/versions.json"

# retrieve available firmware version from versions.json (stripping out whitespace)
hostAvailableVersion=`cat $VERSIONS_JSON | jq -r .$deviceModel.version 2>/dev/null | tr -d '[:space:]'`

if [[ ! -z $hostCurrentVersion ]] && [[ ! -z $hostAvailableVersion ]] && [ "$hostAvailableVersion" != "null" ]; then
    echo ""
    echo "Current MCU version: $hostCurrentVersion"
    echo "Latest MCU version: $hostAvailableVersion"
    echo ""

    if [ -z "$modelOverride" ]; then
        if [ "x$hostCurrentVersion" != "x$hostAvailableVersion" ]; then
            # check for host upgrade forced flag
            hostUpgradeForced=`cat $VERSIONS_JSON | jq -r .$deviceModel.forced 2>/dev/null | tr -d '[:space:]'`
            hostUpgradeFile=$FIRMWARE_BASE_PATH/`cat $VERSIONS_JSON | jq -r .$deviceModel.file 2>/dev/null | tr -d '[:space:]'`

            # if it's forced, start the upgrade immediately
            if [ "$hostUpgradeForced" == "true" ]; then
                if [ -f $AVATAR_UPGRADE_FAILED_FW ] && [ "x$hostAvailableVersion" != "x`cat $AVATAR_UPGRADE_FAILED_FW`" ]; then
                rm -f $AVATAR_UPGRADE_COUNTER $AVATAR_UPGRADE_FAILED_FW
                fi
                [ -f $AVATAR_UPGRADE_COUNTER ] && hostUpgradeCounter=`cat $AVATAR_UPGRADE_COUNTER` || hostUpgradeCounter=0
                if [ "$hostUpgradeCounter" -ge 3 ]; then
                    echo "ERROR: The upgrade tried three times, but the firmware versions do not still match!"
                    echo "$hostAvailableVersion" >$AVATAR_UPGRADE_FAILED_FW
                elif [ -f $hostUpgradeFile ]; then
                    echo $((++hostUpgradeCounter)) >$AVATAR_UPGRADE_COUNTER
                    echo "MCU firmware upgrade is forced, starting immediately!"
                    # disable the hardware watchdog, otherwise the module might be rebooted mid-upgrade
                    echo V > /dev/watchdog
                    hostlink_cli -T 15000 -C update_host 4096 $hostUpgradeFile 2>&1 | tee /tmp/log/host_update
                else
                    echo "ERROR: Wanted to start forced upgrade, but file doesn't exist: $hostUpgradeFile"
                fi
            # otherwise, inform the MCU that an update is available
            else
                echo "MCU firmware upgrade is available, notifying MCU!"

                # State: 0x030B Avatar Upgrade Status
                # Encoding: uint8 Value: 0x02 (Device Upgrade Available)
                hostlink_cli -T 0 -C set_state 0x030B uint8 2
            fi
        else
            echo "MCU firmware is already on the latest version, no need to do anything."
            rm -f $AVATAR_UPGRADE_COUNTER $AVATAR_UPGRADE_FAILED_FW
        fi
    else
        echo "modelOverride is set to $modelOverride; skipping MCU upgrade!"
    fi
fi

