#!/bin/sh
#
# Set different OEM related values in const partition

set -e

# Usage info
show_help()
{
cat << EOF
Usage:
	factory_tool set_const wifi_addr <WIFI-MAC-ADDRESS>
	factory_tool set_const eth_int_addr <ETH1-MAC-ADDRESS>
	factory_tool set_const eth_ext_addr <ETH2-MAC-ADDRESS>
	factory_tool set_const bt_addr <BT-MAC-ADDRESS>
	factory_tool set_const wifiRegulatoryDomain <WIFI-REGION>
	factory_tool set_const googleFactoryLocale <LOCALE>
	factory_tool set_const serial_number <SERIAL-NUMBER>
	factory_tool set_const enc_key <ENCRYPTION-KEY>

Parameters:
	WIFI-MAC-ADDRESS, ETH1-MAC-ADDRESS, ETH2-MAC-ADDRESS, BT-MAC-ADDRESS
		Corresponding MAC addresses. All letters in hexadecimal values
		should be uppercase and each pair should be separated by colon.
		Example: 12:34:56:AB:CD:EF
	WIFI-REGION
		WiFi regulatory domain, where product should be used.
		Alpha-2 code should be used, uppercase, or "00" for world.
		https://en.wikipedia.org/wiki/ISO_3166-2
		Example: US
	LOCALE
		Default locale to be used by Chromecast/GVA.
		For available options contact your system integrator.
		Example: en-US
	SERIAL-NUMBER
		Serial number string (digits, letters, :-_ (4-32 characters)).
		All SUE modules come with a serial from factory (the one from
		the sticker) - but it can be changed to vendor-specific value
		if desired.
	ENCRYPTION-KEY
		Encryption key used to decrypt update (OTA) image. Should be
		used, if image encryption is enabled. Expected input - base64
		encoded string (216 characters together with padding).
		Script swu_enc_key.sh generates proper format.
EOF
}

# Check for correct amount of parameters
if [ "$1" = "help" ]; then
	show_help
	exit 0
elif [ $# -ne 2 ]; then
	echo "Error: wrong amount of parameters"
	show_help
	exit 1
fi

REGEX=""
UPPERCASE=0

case "$1" in
	wifi_addr | \
	eth_int_addr | \
	eth_ext_addr | \
	bt_addr \
	)
		REGEX="^[[:xdigit:]:]\{17\}$"
		UPPERCASE=1
		;;
	wifiRegulatoryDomain)
		REGEX="^[[:alpha:]]\{2\}$\|^[0]\{2\}$"
		UPPERCASE=1
		;;
	googleFactoryLocale)
		REGEX="^[[:lower:]]\{2\}-[[:upper:]]\{2\}[_[:upper:][:digit:]]*$"
		;;
	serial_number)
		REGEX="^[-_:[:alpha:][:digit:]]\{4,32\}$"
		;;
	enc_key)
		REGEX="^[[:alpha:][:digit:]/+=]\{216\}$"
		;;
	*)
		show_help
		exit 1
		;;
esac

# Change all letters  to uppercase
if [ $UPPERCASE -eq 1 ]; then
	VALUE="$(echo "$2" | tr '[:lower:]' '[:upper:]')"
else
	VALUE="$2"
fi

if echo "$VALUE" | grep -o "$REGEX" > /dev/null; then
	fw_setconst "$1" "$VALUE"
	if fw_printconst -n "$1" | grep "^$VALUE$" > /dev/null; then
		echo "$1 was successfully saved"
		exit 0
	else
		echo "Error: $1 was NOT saved"
		exit 1
	fi
else
	echo "Error: wrong format of $1"
	show_help
	exit 1
fi
