#!/bin/sh

# This script checks whether UBI partitions 'nsdk-rootfs' and 'nsdk-settings'
# on the 'data' MTD partition exist, otherwise creates them.

MTD_PART=`cat /proc/mtd | grep '"data"' | sed -e 's/^mtd\([0-9]*\):.*/\1/'`
MTD_SIZE=`cat /sys/class/mtd/mtd${MTD_PART}/size`
SUBPAGESIZE=`cat /sys/class/mtd/mtd${MTD_PART}/subpagesize`

create_partitions()
{
	echo "preinstall: Recreating partitions"

	# caculate settings size
	local SETTINGS_SIZE="8MiB"
	if [ ${MTD_SIZE} -gt 268435456 ]; then
		# 512MB flash
		SETTINGS_SIZE="32MiB"
	fi

	echo "preinstall: Chosen settings size: ${SETTINGS_SIZE}"

	ubimkvol /dev/ubi0 -s ${SETTINGS_SIZE} -N nsdk-settings
	ubimkvol /dev/ubi0 -m -N nsdk-rootfs

	echo "preinstall: Creating partitions complete!"
	exit 128 #128 return value will trigger rescanning of ubi volumes in swupdate
}

format_and_create_partitions()
{
	echo "preinstall: Formatting UBI device"

	ubidetach -m ${MTD_PART}
	ubiformat /dev/mtd${MTD_PART} -y -O ${SUBPAGESIZE}
	ubiattach -m ${MTD_PART} -O ${SUBPAGESIZE}

	# volume is formatted and partitions are created, nothing to do anymore
	echo "preinstall: Formatting complete!"
	create_partitions
}

if [ $(ubinfo | grep 'Count of UBI devices' | sed 's|.*\([0-9]\).*|\1|') -ne 1 ]; then
	echo "preinstall: UBI device not attached"
	# if volume contains junk, ubiattach will fail -> format volume
	# however, if flash is simply empty, it will succeed
	ubiattach -m ${MTD_PART} -O ${SUBPAGESIZE} || format_and_create_partitions
fi

# in factory flash was empty => ubiattach succeeded, but there are no volumes
# yet volume count == 0, therefore we need to just create volumes without
# running time consuming ubiformat
[ $(ubinfo -d 0 -a | grep 'Volumes count' | sed 's|.*\([0-9]\).*|\1|') -eq 0 ] && create_partitions

# next, verify there are all needed volumes are present
# volume count == 2 ?
[ $(ubinfo -d 0 -a | grep 'Volumes count' | sed 's|.*\([0-9]\).*|\1|') -eq 2 ] || format_and_create_partitions
