#!/bin/sh
#
# Play test audio on device

set -e

. /usr/share/factory-tool/scripts/factory_tool_include

VOLUME="$2"

show_help()
{
cat << EOF
Usage:
	factory_tool test_audio_output sine <VOLUME> <DURATION> <FREQUENCY> [OUT_CHANNEL]
	factory_tool test_audio_output pink <VOLUME> <DURATION> [OUT_CHANNEL]
	factory_tool test_audio_output sweep <VOLUME> [OUT_CHANNEL]

Parameters:
	VOLUME
		Volume level.
		Value should be between 0 and 1.
	DURATION
		Duration of specified audio (seconds).
	FREQUENCY
		Frequency of sine wave in Hz.
	OUT_CHANNEL
		Which output channel to use.
		Value should be between 1 and 99 or all (default).
		Value shouldn't exceed maximum channel count.
EOF
}

case "$1" in
	sine)
		if [ $# -ne 5 ] && [ $# -ne 4 ]; then
			echo "Error: wrong amount of parameters"
			show_help
			exit 1
		fi
		# Assign rest of parameters
		DURATION="$3"
		FREQ="$4"
		# Assign "all" value, if OUT_CHANNEL parameter is not provided
		if [ "$5" = "" ]; then
			OUT_CHANNEL="all"
		else
			OUT_CHANNEL="$5"
		fi

		check_channel "$OUT_CHANNEL" "$MAX_OUTPUT_CH"

		MIX_MATRIX=`generate_channel_mix_matrix 1 "$MAX_OUTPUT_CH" 1 "$OUT_CHANNEL"`

		gst-launch-1.0 audiotestsrc freq=$FREQ is-live=true volume=$VOLUME ! \
			"audio/x-raw,format=S32LE,rate=48000,channels=1" ! \
			audioconvert mix-matrix="$MIX_MATRIX" ! \
			"audio/x-raw,format=S32LE,rate=48000,channels=$MAX_OUTPUT_CH" ! \
			autoaudiosink &
		sleep "$DURATION"
		# Kill background processes after sleep
		pkill -P $$
		;;

	pink)
		if [ $# -ne 4 ] && [ $# -ne 3 ]; then
			echo "Error: wrong amount of parameters"
			show_help
			exit 1
		fi
		# Assign rest of parameters
		DURATION="$3"
		# Assign "all" value, if OUT_CHANNEL parameter is not provided
		if [ "$4" = "" ]; then
			OUT_CHANNEL="all"
		else
			OUT_CHANNEL="$4"
		fi

		check_channel "$OUT_CHANNEL" "$MAX_OUTPUT_CH"

		MIX_MATRIX=`generate_channel_mix_matrix 1 "$MAX_OUTPUT_CH" 1 "$OUT_CHANNEL"`

		gst-launch-1.0 audiotestsrc wave=6 is-live=true volume=$VOLUME ! \
			"audio/x-raw,format=S32LE,rate=48000,channels=1" ! \
			audioconvert mix-matrix="$MIX_MATRIX" ! \
			"audio/x-raw,format=S32LE,rate=48000,channels=$MAX_OUTPUT_CH" ! \
			autoaudiosink &
		sleep "$DURATION"
		# Kill background processes after sleep
		pkill -P $$
		;;

	sweep)
		if [ $# -ne 3 ] && [ $# -ne 2 ]; then
			echo "Error: wrong amount of parameters"
			show_help
			exit 1
		fi
		# Assign "all" value, if OUT_CHANNEL parameter is not provided
		if [ "$3" = "" ]; then
			OUT_CHANNEL="all"
		else
			OUT_CHANNEL="$3"
		fi

		check_channel "$OUT_CHANNEL" "$MAX_OUTPUT_CH"

		MIX_MATRIX=`generate_channel_mix_matrix 1 "$MAX_OUTPUT_CH" 1 "$OUT_CHANNEL"`

		gst-launch-1.0 filesrc location=/usr/share/factory-tool/files/sweep-20-22k-10s.ogg ! \
			oggdemux ! vorbisdec ! audioconvert ! "audio/x-raw,format=S32LE,rate=48000,channels=1" ! \
			audioconvert mix-matrix="$MIX_MATRIX" ! "audio/x-raw,format=S32LE,rate=48000,channels=$MAX_OUTPUT_CH" ! \
			volume volume=$VOLUME ! audioresample ! autoaudiosink
		;;

	*)
		show_help
		;;
esac

exit 0
