$(document).ready(function () {
  const maxVolId = "#maximum-volume";
  const maxOnVolId = "#maximum-on-volume";

  startup();

  function startup() {
    restorePreviousValues();
    $(maxVolId).on("input", maximumVolumeChanged);
    $(maxOnVolId).on("input", maximumOnVolumeChanged);

    window.socketMessageReceived = function(msg) {
      if (msg[2] == 0x1D) {
        window.webSocket.send(0x29, [0xF0]);
      } else if (msg[2] == 0x29) {
        updateValuesUsing(parseGeneralSetupBuffer(msg));
      } else if (msg[2] === 0x43) {
        updateAudioFormat(msg[5], msg[6]);
      } else if (msg[2] === 0x44) {
        updateSampleRate(msg[5]);
      }
    };

    window.webSocket.send(0x29, [0xF0]);
  }

  function updateAudioFormat(stream, channels) {
    const model = updateModel("incoming-format", Utils.audioFormat(stream, channels));
  }

  function updateSampleRate(rate) {
    const model = updateModel("incoming-sample-rate", Utils.parseSampleRate(rate));
    updateValuesUsing(model);
  }

  function parseGeneralSetupBuffer(buffer) {
    if (buffer.length != 38) {
      throw "Invalid buffer length: " + buffer.length;
    }

    if (!validateCommandCode(buffer, 0x29)) {
      throw "Invalid command code.";
    }

    var data = buffer.slice(5, buffer.length - 1);

    return {
      "source-input": Utils.asciiFromBuffer(data, { min: 0, max: 10 }),
      "incoming-format": Utils.audioFormat(data[10], data[11]),
      "incoming-sample-rate": Utils.parseSampleRate(data[12]),
      "incoming-bitrate": Utils.parseBitrate(data[13]),
      "dialnorm": Utils.parseDialNorm(data[14]),
      "incoming-video-resolution": Utils.parseIncomingVideoResolution(data.slice(17,21)),
      "audio-compression": data[23],
      "balance": Utils.parseSignedValue(data[24]),
      "dts-dialogue-control": data[25],
      "maximum-volume": data[26],
      "maximum-on-volume": data[27],
      "display-on-time": data[28],
      "control-options": data[29],
      "power-on": data[30],
      "language": data[31],
    };
  }

  window.encodeModel = function(model) {
    var data = new Array(32).fill(0);
    data[23] = model["audio-compression"];
    data[24] = Utils.toSignedValue(model["balance"]);
    data[25] = model["dts-dialogue-control"];
    data[26] = model["maximum-volume"];
    data[27] = model["maximum-on-volume"];
    data[28] = model["display-on-time"];
    data[29] = model["control-options"];
    data[30] = model["power-on"];
    data[31] = model["language"];
    return { code: 0x29, data: data };
  };

  function maximumVolumeChanged() {
    var maxVol = $(maxVolId).val();
    var maxOnVol = $(maxOnVolId).val();

    if (maxVol < maxOnVol) {
      $(maxOnVolId).val(maxVol);
    }
  }

  function maximumOnVolumeChanged() {
    var maxVol = $(maxVolId).val();
    var maxOnVol = $(maxOnVolId).val();

    if (maxOnVol > maxVol) {
      $(maxOnVolId).val(maxVol);
    }
  }

  window.shouldRespondToStepper = function(key, modifier) {
    if (key === "maximum-on-volume" && modifier === "increase") {
      return $(maxOnVolId).val() < $(maxVolId).val();
    }

    return true;
  };
});
