$(document).ready(function() {
  startup();

  function startup() {
    restorePreviousValues();
    setupCallbacks();

    updateMSBLabel(parseInt($("#units").val(), 10));

    if (parseInt($("#units").val(), 10) == 1) {
      convertDefaultFloatValuesToIntegers();
    }

    window.socketMessageReceived = function(msg) {
      if (msg[2] == 0x2B) {
        updateUnitsPicker(msg[5]);
        updateValuesUsing(parseSpeakerDistancesBuffer(msg));
        updateMSBLabel(parseInt($("#units").val(), 10));
      }

      if (msg[2] == 0x2A) {
        disableUnsupportedSpeakers(msg);
      }
    };

    window.webSocket.send(0x2B, [0xF0]);
    window.webSocket.send(0x2A, [0xF0]);
  }

  function updateUnitsPicker(units) {
    var options = '';

    if (units === 2) {
      options = '<option value="2">ms</option>';
    } else {
      options = '<option value="0">Metres</option><option value="1">Feet</option>';
    }

    $("#units").empty().append(options);
  }

  function setupCallbacks() {
    $("input[type=number]").on('change', function() {
      const units = parseInt($("#units").val(), 10);
      if (units === 0) {
        const rounded = roundCM($(this).val());
        $(this).val(rounded);
      } else if (units === 1) {
        const rounded = roundImperial($(this));
        $(this).val(rounded);
      } else {
        const rounded = roundMS($(this).val());
        $(this).val(rounded);
      }

      var model = updateModel($(this).attr('id'), $(this).val());
      sendModel(model);
    });
  }

  function parseSpeakerDistancesBuffer(buffer) {
    if (buffer.length != 39) {
      throw "Invalid buffer length: " + buffer.length;
    }

    if (!validateCommandCode(buffer, 0x2B)) {
      throw "Invalid command code.";
    }

    var data = buffer.slice(5, buffer.length - 1);

    if (data[0] == 1) {
      return parseImperialDistances(data);
    } else {
      return parseMetricDistances(data);
    }
  }

  function parseMetricDistances(data) {
    return {
      "units": data[0],
      "front-left-msb": parseMetricSlice(data.slice(1, 3)),
      "centre-msb": parseMetricSlice(data.slice(3, 5)),
      "front-right-msb": parseMetricSlice(data.slice(5, 7)),
      "surr-right-msb": parseMetricSlice(data.slice(7, 9)),
      "surr-back-right-msb": parseMetricSlice(data.slice(9, 11)),
      "surr-back-left-msb": parseMetricSlice(data.slice(11, 13)),
      "surr-left-msb": parseMetricSlice(data.slice(13, 15)),
      "left-top-front-msb": parseMetricSlice(data.slice(15, 17)),
      "right-top-front-msb": parseMetricSlice(data.slice(17, 19)),
      "left-top-back-msb": parseMetricSlice(data.slice(19, 21)),
      "right-top-back-msb": parseMetricSlice(data.slice(21, 23)),
      "subwoofer-msb": parseMetricSlice(data.slice(23, 25)),
      "channel-13-msb": parseMetricSlice(data.slice(25, 27)),
      "channel-14-msb": parseMetricSlice(data.slice(27, 29)),
      "channel-15-msb": parseMetricSlice(data.slice(29, 31)),
      "channel-16-msb": parseMetricSlice(data.slice(31, 33))
    };
  }

  function parseMetricSlice(value) {
    return value[0].toString() + "." + pad(value[1]);
  }

  function pad(num) {
    num = num.toString();
    if (num.length < 2) {
      num = "0" + num;
    }
    return num;
  }

  function disableUnsupportedSpeakers(buffer) {
    if (buffer.length != 19) {
      return;
    }

    $(".front-speaker-group :enabled").attr('disabled', buffer[5] === 16);
    $(".centre-speaker-group :enabled").attr("disabled", buffer[6] === 16);
    $(".surr-speaker-group :enabled").attr("disabled", buffer[7] === 16);
    $(".surr-back-speaker-group :enabled").attr("disabled", buffer[8] === 16);
    $(".top-front-speaker-group :enabled").attr("disabled", buffer[9] === 16);
    $(".top-back-speaker-group :enabled").attr("disabled", buffer[10] === 16);
    $(".subwoofer-speaker-group :enabled").attr("disabled", buffer[11] === 1);
    $(".channel-13-14-speaker-group :enabled").attr("disabled", buffer[12] === 17);
    $(".channel-15-16-speaker-group :enabled").attr("disabled", buffer[13] === 33);
  }

  window.encodeModel = function(model) {
    var units = model["units"];
    var data = [units];
    var speakers = [
      "front-left",
      "centre",
      "front-right",
      "surr-right",
      "surr-back-right",
      "surr-back-left",
      "surr-left",
      "left-top-front",
      "right-top-front",
      "left-top-back",
      "right-top-back",
      "subwoofer",
      "channel-13",
      "channel-14",
      "channel-15",
      "channel-16"
    ].map(s => encodeSpeaker(s, model));
    return { "code": 0x2B, "data": data.concat(speakers).flat() };
  };

  function encodeSpeaker(speaker, model) {
    const msb = speaker + "-msb";
    const isImperial = model["units"] === 1;
    if (isImperial) {
      var lsb = speaker + "-lsb";
      return [model[msb], model[lsb]];
    } else {
      return model[msb].split('.').map(n => Number(n));
    }
  }

  function parseImperialDistances(data) {
    return {
      "units": data[0],
      "front-left-msb": data[1],
      "front-left-lsb": data[2],
      "centre-msb": data[3],
      "centre-lsb": data[4],
      "front-right-msb": data[5],
      "front-right-lsb": data[6],
      "surr-right-msb": data[7],
      "surr-right-lsb": data[8],
      "surr-back-right-msb": data[9],
      "surr-back-right-lsb": data[10],
      "surr-back-left-msb": data[11],
      "surr-back-left-lsb": data[12],
      "surr-left-msb": data[13],
      "surr-left-lsb": data[14],
      "left-top-front-msb": data[15],
      "left-top-front-lsb": data[16],
      "right-top-front-msb": data[17],
      "right-top-front-lsb": data[18],
      "left-top-back-msb": data[19],
      "left-top-back-lsb": data[20],
      "right-top-back-msb": data[21],
      "right-top-back-lsb": data[22],
      "subwoofer-msb": data[23],
      "subwoofer-lsb": data[24],
      "channel-13-msb": data[25],
      "channel-13-lsb": data[26],
      "channel-14-msb": data[27],
      "channel-14-lsb": data[28],
      "channel-15-msb": data[29],
      "channel-15-lsb": data[30],
      "channel-16-msb": data[31],
      "channel-16-lsb": data[32]
    };
  }

  $("#units").change(function() {
    var units = parseInt($(this).val(), 10);
    var isMetric = units == 0;
    updateModel("units", Number(units));
    updateMSBLabel(units);
    convertAllValues(isMetric);

    var model = JSON.parse(sessionStorage.getItem(window.currentPage));
    sendModel(model);
  });

  function updateMSBLabel(units) {
    var max;

    if (units == 0x00) {
      max = "13.60";
      $(".msb-label").text("m");
      $(".lsb").hide();
    } else if (units == 0x01) {
      max = "40";
      $(".msb-label").text("ft");
      $(".lsb").show();
    } else {
      max = "46.00";
      $(".msb-label").text("ms");
      $(".lsb").hide();
    }

    $("input[id$=msb]").attr({
      "max": max,
    });
  }

  function convertAllValues(isMetric) {
    const msbFields = $("input[id$=msb]");
    const lsbFields = $("input[id$=lsb]");

    for (var i = 0; i < msbFields.length; i++) {
      const msb = Number(msbFields[i].value) || 0;
      const lsb = Number(lsbFields[i].value) || 0;
      var newValues;

      if (isMetric) {
        newValues = convertFeetToMeters([msb, lsb]);
      } else {
        newValues = convertMetersToFeetInches(msb);
      }

      msbFields.eq(i).val(newValues[0]);
      updateModel(msbFields.eq(i).attr('id'), newValues[0]);
      lsbFields.eq(i).val(newValues[1]);
      updateModel(lsbFields.eq(i).attr('id'), newValues[1]);
    }
  }

  function convertMetersToFeetInches(input) {
    // Should be 0.0254, but the AVR rounds it down so match the behaviour
    const totalInches = Math.round(input / 0.025);
    const feet = totalInches / 12;
    const inches = (feet % 1) * 12;

    return [Math.floor(feet), Math.round(inches)];
  }

  function convertFeetToMeters(input) {
    const msb = input[0];
    const lsb = input[1];
    const totalInches = (msb * 12) + lsb;
    // Should be 0.0254, but the AVR rounds it down so match the behaviour
    const metres = roundCM(totalInches * 0.025);

    return [metres, 0];
  }

  // The AVR uses its own rounding mechanism so match that
  function roundCM(input) {
    // clamp values between 0–13.60m
    input = Math.min(Math.max(input, 0), 13.60);
    const flooredInput = Math.floor(input);
    const inputRemainder = (input - flooredInput) * 10;
    const flooredRemainder = Math.floor(inputRemainder);
    const shifted = (inputRemainder - flooredRemainder) * 10;
    var rounded = 0;

    if (shifted < 2) {
      rounded = 0;
    } else if (shifted < 4) {
      rounded = 3;
    } else if (shifted < 6.5) {
      rounded = 5;
    } else if (shifted < 8.5) {
      rounded = 8;
    } else {
      rounded = 10;
    }

    const tenths = flooredRemainder / 10;
    const hundredths = rounded / 100;

    return Number.parseFloat(flooredInput + hundredths + tenths).toFixed(2);
  }

  function roundImperial(element) {
    const value = parseInt(element.val());
    if (element.attr('id').includes('msb')) {
      return Math.min(Math.max(value, 0), 46);
    } else {
      return Math.min(Math.max(value, 0), 11);
    }
  }

  function roundMS(input) {
    var clamped = Math.min(Math.max(input, 0), 46.00);
    return Number.parseFloat(clamped).toFixed(2);
  }

  function convertDefaultFloatValuesToIntegers() {
      $("input[id$=msb]").each(function() {
        if ($(this).val() == 0.0) {
          $(this).val("0");
        }
      });
  }

});
