$(document).ready(function() {
  startup();

  function startup() {
    restorePreviousValues();
    registerCallbacks();

    window.socketMessageReceived = function(msg) {
      if (msg[2] == 0x2C) {
        updateValuesUsing(parseSpeakerLevelsBuffer(msg));
        $("#test-tone-source").change();
      }

      if (msg[2] == 0x2A) {
        disableUnsupportedSpeakers(msg);
      }
    };

    window.webSocket.send(0x2C, [0xF0]);
    window.webSocket.send(0x2A, [0xF0]);
  }

  function registerCallbacks() {
    $("#test-tone-source").change(function() {
      const isDisabled = Number($(this).val()) === 1;
      updateSpeakerNoiseButtonAvailability(isDisabled);
    });

    $("#test-tone-source").change();

    $("button.speaker-button").on('click', function() {
      const icon = $(this).children().first();
      $('button>.fa-stop').not(icon).removeClass('fa-stop').addClass('fa-volume-up');

      if (icon.is('.fa-stop')) {
        icon.removeClass('fa-stop').addClass('fa-volume-up');
        generateNoise(null);
      } else {
        icon.removeClass('fa-volume-up').addClass('fa-stop');
        var speaker = $(this).attr('noise-output');
        generateNoise(speaker);
      }
    });

    $("button.volume").on('click', function() {
      handleVolumeButton($(this));
    });
  }

  function updateSpeakerNoiseButtonAvailability(isGloballyDisabled) {
    if (isGloballyDisabled) {
      $("button.speaker-button").attr("disabled", isGloballyDisabled);
    } else {
      const buttons = $("button.speaker-button");
      buttons.each(function(index, item) {
        const isDisabled = $(item).parent().parent().find('input').attr('disabled');
        $(item).attr('disabled', isDisabled);
      });
    }
  }

  function handleVolumeButton(button) {
    const id = button.attr('id').split('-');
    const direction = id.pop();
    const key = id.join('-');
    var currentValue = getValueFor(key);
    if (direction === "volup") {
      currentValue += 0.5;
    } else {
      currentValue -= 0.5;
    }

    if (currentValue < -10 || currentValue > 10) {
      return;
    }

    const model = updateModel(key, Number(currentValue));
    sendModel(model);
    updateValuesUsing(model);
  }

  function generateNoise(channel) {
    var model = updateModel("noise-source", Number(channel));
    sendModel(model);
  }

  function parseSpeakerLevelsBuffer(buffer) {
    if (buffer.length != 24) {
      throw "Invalid buffer length: " + buffer.length;
    }

    if (!validateCommandCode(buffer, 0x2C)) {
      throw "Invalid command code.";
    }

    var data = buffer.slice(5, buffer.length - 1);

    return {
      "test-tone-source": data[0],
      "front-left": Utils.parseVolumeLevel(data[1]),
      "centre": Utils.parseVolumeLevel(data[2]),
      "front-right": Utils.parseVolumeLevel(data[3]),
      "surr-right": Utils.parseVolumeLevel(data[4]),
      "surr-back-right": Utils.parseVolumeLevel(data[5]),
      "surr-back-left": Utils.parseVolumeLevel(data[6]),
      "surr-left": Utils.parseVolumeLevel(data[7]),
      "left-top-front": Utils.parseVolumeLevel(data[8]),
      "right-top-front": Utils.parseVolumeLevel(data[9]),
      "left-top-back": Utils.parseVolumeLevel(data[10]),
      "right-top-back": Utils.parseVolumeLevel(data[11]),
      "subwoofer": Utils.parseVolumeLevel(data[12]),
      "channel-13": Utils.parseVolumeLevel(data[13]),
      "channel-14": Utils.parseVolumeLevel(data[14]),
      "channel-15": Utils.parseVolumeLevel(data[15]),
      "channel-16": Utils.parseVolumeLevel(data[16]),
      "noise-source": data[17]
    };
  }

  window.encodeModel = function(model) {
    var data = new Array(17).fill(0);
    data[0]  = model["test-tone-source"];
    data[1]  = Utils.encodeVolumeLevel(model["front-left"]);
    data[2]  = Utils.encodeVolumeLevel(model["centre"]);
    data[3]  = Utils.encodeVolumeLevel(model["front-right"]);
    data[4]  = Utils.encodeVolumeLevel(model["surr-right"]);
    data[5]  = Utils.encodeVolumeLevel(model["surr-back-right"]);
    data[6]  = Utils.encodeVolumeLevel(model["surr-back-left"]);
    data[7]  = Utils.encodeVolumeLevel(model["surr-left"]);
    data[8]  = Utils.encodeVolumeLevel(model["left-top-front"]);
    data[9]  = Utils.encodeVolumeLevel(model["right-top-front"]);
    data[10] = Utils.encodeVolumeLevel(model["left-top-back"]);
    data[11] = Utils.encodeVolumeLevel(model["right-top-back"]);
    data[12] = Utils.encodeVolumeLevel(model["subwoofer"]);
    data[13] = Utils.encodeVolumeLevel(model["channel-13"]);
    data[14] = Utils.encodeVolumeLevel(model["channel-14"]);
    data[15] = Utils.encodeVolumeLevel(model["channel-15"]);
    data[16] = Utils.encodeVolumeLevel(model["channel-16"]);
    data[17] = model["noise-source"];
    return { code: 0x2C, data };
  };

  function disableUnsupportedSpeakers(buffer) {
    if (buffer.length != 19) {
      return;
    }

    $(".front-speaker-group :enabled").attr('disabled', buffer[5] === 16);
    $(".centre-speaker-group :enabled").attr("disabled", buffer[6] === 16);
    $(".surr-speaker-group :enabled").attr("disabled", buffer[7] === 16);
    $(".surr-back-speaker-group :enabled").attr("disabled", buffer[8] === 16);
    $(".top-front-speaker-group :enabled").attr("disabled", buffer[9] === 16);
    $(".top-back-speaker-group :enabled").attr("disabled", buffer[10] === 16);
    $(".subwoofer-speaker-group :enabled").attr("disabled", buffer[11] === 1);
    $(".channel-13-14-speaker-group :enabled").attr("disabled", buffer[12] === 17);
    $(".channel-15-16-speaker-group :enabled").attr("disabled", buffer[13] === 33);
  }
});
