#!/usr/bin/env python3

import array
import struct
import uuid
import os
from cryptography.hazmat.backends import default_backend
from cryptography.hazmat.primitives import hashes
from cryptography.hazmat.primitives import padding
from cryptography.hazmat.primitives import serialization
from cryptography.hazmat.primitives.ciphers import Cipher
from cryptography.hazmat.primitives.ciphers import algorithms
from cryptography.hazmat.primitives.ciphers import modes
from cryptography.hazmat.primitives.asymmetric.padding import PKCS1v15

def addToBuffer(buff, rsa_long):
	arr = bytearray()
	while rsa_long:
		arr = struct.pack("<B", rsa_long % 256) + arr
		rsa_long = rsa_long >> 8
	buff += bytearray(struct.pack('<I', len(arr)))
	for x in arr:
		buff += bytearray(struct.pack("B", x))

def serializeRsaKey(key):
	buff = bytearray()

	addToBuffer(buff, key.public_numbers.n)
	addToBuffer(buff, key.public_numbers.e)
	addToBuffer(buff, key.d)
	addToBuffer(buff, key.p)
	addToBuffer(buff, key.q)
	addToBuffer(buff, key.dmp1)
	addToBuffer(buff, key.dmq1)
	addToBuffer(buff, key.iqmp)
	return buff

def get_args():
	from argparse import ArgumentParser

	parser = ArgumentParser()
	parser.add_argument('--key', required=True, help='master key used for encryption (PEM format)')

	group = parser.add_mutually_exclusive_group(required=True)
	group.add_argument('--inKey', help='private RSA key that will be encrypted (PEM format)')
	group.add_argument('--inData', help='data that will be encrypted')

	parser.add_argument('--out', type=str, required=True, help='destination path for encrypted RSA key')
	parser.add_argument('--ta', type=str, required=False, help='TA uuid')
	return parser.parse_args()

def main():
	args = get_args()
	pub_key = None
	data_to_encrypt = None
	ta_uuid = None

	with open(args.key, 'rb') as f:
		pub_key = serialization.load_pem_public_key(f.read(), backend=default_backend())

	if args.inKey:
		with open(args.inKey, 'rb') as f:
			prv_key = serialization.load_pem_private_key(f.read(), password=None, backend=default_backend())
			data_to_encrypt = serializeRsaKey(prv_key)

	if args.inData:
		with open(args.inData, 'rb') as f:
			data_to_encrypt = f.read()

	if args.ta:
		ta_uuid = uuid.UUID(args.ta)

	# generate random keys
	aes_iv = os.urandom(16)
	aes_key = os.urandom(32)

	derived_aes_key = aes_key
	if ta_uuid:
		sha256 = hashes.Hash(hashes.SHA256(), backend=default_backend())
		sha256.update(aes_key)
		sha256.update(ta_uuid.bytes_le)
		derived_aes_key = sha256.finalize()

	aes_enc = Cipher(algorithms.AES(derived_aes_key), modes.CBC(aes_iv), default_backend()).encryptor()

	# pkcs 7
	padder = padding.PKCS7(algorithms.AES.block_size).padder()
	data_to_encrypt = padder.update(bytes(data_to_encrypt)) + padder.finalize()

	outf = open(args.out, 'wb+')
	outf.write(aes_iv)
	outf.write(pub_key.encrypt(aes_key, PKCS1v15()))
	outf.write(aes_enc.update(bytes(data_to_encrypt)) + aes_enc.finalize())
	outf.close()

if __name__ == "__main__":
	main()
