#!/bin/sh
#
# Check consistency of swufit (recovery image)

set -e

# Location of public key (assumed to be always there)
PUBKEY_PATH="/home/root/verified-boot.dtb"
# Check patterns
KERNEL_PATTERN=" *Type: *Kernel Image"
FDT_PATTERN=" *Type: *Flat Device Tree"
RAMDISK_PATTERN=" *Type: *RAMDisk Image"
SIGNATURE_PATTERN="Signature check OK"

if [ $# -eq 0 ]; then
	if [ "$(cat /proc/mtd | grep swufit | wc -l)" -ne "1" ]; then
		echo "Error: Unable to locate swufit partition"
		exit 1
	fi

	SWU_MTD_PART="/dev/$(cat /proc/mtd | grep swufit | cut -f1 -d:)"
	FIT_FILE_PATH="/tmp/swufit"

	# We need to use the `nanddump` tool to read the `swufit` partition because we
	# need to be able to handle bad blocks using the same mechanism as U-Boot does
	# (in this case skipping them).
	nanddump --bb=skipbad -f "$FIT_FILE_PATH" "$SWU_MTD_PART" > /dev/null 2>&1

	# Separate error handling
	set +e
	# stderr is forwarded to stdout, since "Signature check OK" message
	# for some reason is outputted to stderr
	OUTPUT="$(fit_check_sign -f "$FIT_FILE_PATH" -k "$PUBKEY_PATH" 2>&1)"
	EXIT_CODE=$?
	set -e

	rm -f "$FIT_FILE_PATH"

	# Check that each pattern appears at least once
	if [ "$EXIT_CODE" -eq 0 ] && \
		echo "$OUTPUT" | grep -xq "$KERNEL_PATTERN" && \
		echo "$OUTPUT" | grep -xq "$FDT_PATTERN" && \
		echo "$OUTPUT" | grep -xq "$RAMDISK_PATTERN" && \
		echo "$OUTPUT" | grep -xq "$SIGNATURE_PATTERN"; then
		echo "Success: swufit partition consistency check PASSED"
		exit 0
	else
		echo "Error: swufit partition consistency check FAILED"
		# To add some extra info, when check fails
		echo "$OUTPUT"
		exit 1
	fi
else
	# If extra arguments are provided
	echo "Error: Command doesn't expect any parameters"
	exit 1
fi
