$(document).ready(function() {
  startup();

  function startup() {
    restorePreviousValues();
    registerButtonHandling();
    registerC4Handler();
    $("#c4-sddp").change();
    toggleDANTEVisibility();

    window.socketMessageReceived = function(msg) {
      if (msg[2] == 0x33) {
        updateValuesUsing(parseEngineeringMenuBuffer(msg));
        $("#c4-sddp").change();
      }
    };

    $("#region").on("change", sendRegionChange);

    window.webSocket.send(0x33, [0xF0]);
  }

  function parseEngineeringMenuBuffer(buffer) {
    if (buffer.length != 57) {
      throw "Invalid buffer length: " + buffer.length;
    }

    if (!validateCommandCode(buffer, 0x33)) {
      throw "Invalid command code.";
    }

    var data = buffer.slice(5, buffer.length - 1);

    return {
      "region": data[10],
      "remote-code": data[11],
      "standby-mode": data[12],
      "protection-sensitivity": data[13],
      "use-display-hdmi": data[14],
      "display-type": data[15],
      "dante": data[16],
      "c4-sddp": data[17],
      "c4-identify": data[18],
      "shutdown-code": shutdownCode(data[19]),
      "host-version": Utils.asciiFromBuffer(data, {"min": 20, "max": 29} )
    };
  }

  window.encodeModel = function(model) {
    var data = new Array(18).fill(0);
    data[10] = model["region"];
    data[11] = model["remote-code"];
    data[12] = model["standby-mode"];
    data[13] = model["protection-sensitivity"];
    data[14] = model["use-display-hdmi"];
    data[15] = model["display-type"];
    data[16] = model["dante"];
    data[17] = model["c4-sddp"];
    data[18] = model["c4-identify"];
    return { code: 0x33, data: data };
  };

  function registerButtonHandling() {
    const timeout = 5000;
    $("#reset-factory-defaults").click(function() {
      if (confirm("Are you sure you want to reset to factory defaults?")) {
        var text = "Restoring factory defaults...";
        showAlert(text, $(this), timeout);
        sendButtonPress(0);
      }
    });

    $("#update").click(function() {
      var text = "Checking for updates...";
      showAlert(text, $(this), timeout);
        sendButtonPress(1);
    });

    $("#restore-secure-backup").click(function() {
      if (confirm("Are you sure you want to restore secure backup?")) {
        var text = "Restoring secure backup...";
        showAlert(text, $(this), timeout);
        sendButtonPress(2);
      }
    });

    $("#store-secure-backup").click(function() {
      const pin = prompt("Enter 4-digit PIN", "0000").trim();
      if (pin.match(/^[0-9]{4}$/)) {
        showAlert("Storing secure backup...", $(this), timeout);
        storeSecureBackup(pin);
      } else {
        showAlert("PIN must be 4-digits between 0000-9999.", $(this), timeout, "alert-danger");
      }
    });

    $("#restore-usb-backup").click(function() {
      if (confirm("Are you sure you want to restore USB backup?")) {
        var text = "Restoring USB backup...";
        showAlert(text, $(this), timeout);
        sendButtonPress(4);
      }
    });

    $("#store-usb-backup").click(function() {
      var text = "Storing USB backup...";
      showAlert(text, $(this), timeout);
        sendButtonPress(5);
    });

    $("#c4-identify").click(function() {
      var text = "Sending Control4 Identify";
      showAlert(text, $(this), timeout);
        sendButtonPress(18);
    });
  }

  function shutdownCode(code) {
    switch (code) {
      case 0:
      return "00 - normal";
      case 1:
      return "01 - amp DC offset";
      case 2:
      return "02 - amp over temp";
      case 3:
      return "03 - amp over current";
    }
  }

  function sendButtonPress(location) {
    var model = getCurrentModel();
    model.data[location] = 1;
    sendEncodedModel(model);
  }

  function storeSecureBackup(pin) {
    if (pin.length != 4) { return; }

    const pinBuffer = pin.split('').map(e => Number(e));
    var model = getCurrentModel();
    model.data[3] = 1; // Set Store Backup byte
    model.data.splice(6, pinBuffer.length, ...pinBuffer); // Insert the pin array
    sendEncodedModel(model);
  }

  function getCurrentModel() {
    var model = JSON.parse(sessionStorage.getItem("engineering"));
    return window.encodeModel(model);
  }

  function sendEncodedModel(model) {
    window.webSocket.send(model.code, model.data);
  }

  function registerC4Handler() {
    $("#c4-sddp").change(function() {
      var isDisabled = $(this).val() == 0;
      $("#c4-identify").prop("disabled", isDisabled);
    });
  }

  function toggleDANTEVisibility() {
    if (getDeviceName() != "JBL SYNTHESIS") {
      $(".dante-field").hide();
    }
  }

  function sendRegionChange() {
    const key = $(this).attr("id");
    const value = $(this).val();
    const model = updateModel(key, value);
    localStorage.setItem("regionCode", value);
    sendModel(model);
    showRegionChangeMessage();
    setTimeout(window.webSocket.disconnect, 1000);
    setTimeout(window.location.reload.bind(window.location), 40000);
  }

  function showRegionChangeMessage() {
    const html = `
      <h1>Changing region</h1>
      <p class="d-block">This page will automatically refresh once the region has been changed.</p>
    `;
    $("#fsol-warning>.message").html(html);

    const spinner = `
    <div class="spinner-border text-primary" role="status">
      <span class="sr-only">Restarting...</span>
    </div>
    `;
    $("#fsol-warning>.icon").html(spinner);
    $("#menu-open-overlay").fadeIn(100);
  }
});
