"use strict";

function pad02(i) {
	return ((i <= 9) ? "0" : "") + i.toString();
}

function msec2time(time) {
	time = parseInt(time / 1000)
	var ret = pad02(parseInt(time / 60) % 60) + ":" + pad02(time % 60)
	time = parseInt(time / 3600)
	if (time > 0)
		ret = pad02(parseInt(time)) + ":" + ret
	return ret
}

function checkPlaymodeSupported(shuffle, repeat, playModeSupported, playModeStrings) {
	var playModeCode = repeat * 2 + shuffle;

	if (playModeSupported[playModeStrings[playModeCode]])
		return true;
	else
		return false;
}

function PlayerController(browserRef) {
	var controlPath = null;
	var controlPlayModePath = null;
	var dataPath = null;
	var data_playTimePath = null;
	var data_volumePath = null;
	var data_mutePath = null;
	var self = this;

	var onDataChangeHandlers = []
	var onProgressChangeHandlers = []
	var onVolumeChangeHandlers = []
	var onMuteChangeHandlers = []
	var onPlayModeChangeHandlers = []

	var browser = browserRef;

	this.ui = {
		'title': "",
		'artist': "",
		'album': "",
		'codec': "",
		'mqa': "",
		'audioFormat': "",
		'icon': "",

		'volume': 0,
		'playTime': 0,
		'playTimeTotal': -1,
		'playId': null,
		'contextPath': null,
		'state': "stopped",
		'error': null,

		'playmode': {
			'shuffle': false,
			'repeat': false,
			'playModeSupported': [{
					"normal": false
				},
				{
					"shuffle": false
				},
				{
					"repeatOne": false
				},
				{
					"shuffleRepeatOne": false
				},
				{
					"repeatAll": false
				},
				{
					"shuffleRepeatAll": false
				}
			]
		},

		'buttons': {
			'shuffle': 0,
			'repeat': 0,
			'next': false,
			'previous': false,
			'pause': false,
			/* Play and stop buttons shall always be available. TODO: add to docs */
		},

		'seekTime': false,
	};

	var playModeStrings = [
		"normal",
		"shuffle",
		"repeatOne",
		"shuffleRepeatOne",
		"repeatAll",
		"shuffleRepeatAll"
	];

	this.mediaData = null;

	this.registerDataChangeHandler = function(h) {
		onDataChangeHandlers.push(h)
		h(self.ui)
	}

	this.registerProgressChangeHandler = function(h) {
		onProgressChangeHandlers.push(h)
		h(self.ui)
	}

	this.registerPlayModeChangeHandler = function(h) {
		onPlayModeChangeHandlers.push(h)
		h(self.ui)
	}

	this.registerVolumeChangeHandler = function(h) {
		onVolumeChangeHandlers.push(h)
		h(self.ui)
	}

	this.registerMuteChangeHandler = function(h) {
		onMuteChangeHandlers.push(h)
		h(self.ui)
	}

	this.onPlayModeChange = function(page, state) {
		var value = page.get("value") || {};

		if (value.playerPlayMode == undefined) value.playerPlayMode = {}

		if ((value.playerPlayMode == "repeatOne") || (value.playerPlayMode == "shuffleRepeatOne")) {
			self.ui.buttons.repeat = 1;
		} else if ((value.playerPlayMode == "repeatAll") || (value.playerPlayMode == "shuffleRepeatAll")) {
			self.ui.buttons.repeat = 2;
		} else {
			self.ui.buttons.repeat = 0;
		}

		if ((value.playerPlayMode == "shuffle") || (value.playerPlayMode == "shuffleRepeatAll") || (value.playerPlayMode == "shuffleRepeatOne")) {
			self.ui.buttons.shuffle = 1;
		} else {
			self.ui.buttons.shuffle = 0;
		}

		for (var i in onPlayModeChangeHandlers) {
			onPlayModeChangeHandlers[i](self.ui)
		}
	}

	this.onPlayTimeChange = function(page, state) {
		var value = 0;
		if (page.get("value") != null) {
			var value = getTypedValue(page.get("value"))
			if (isNaN(parseInt(value)))
				value = 0
		}
		self.ui.playTime = value
		for (var i in onProgressChangeHandlers)
			onProgressChangeHandlers[i](self.ui)
	}

	this.onVolumeChange = function(page, state) {
		var value = 0;
		if (page.get("value") != null) {
			var value = getTypedValue(page.get("value"))
			if (isNaN(parseInt(value)))
				value = 0
		}
		self.ui.volume = value
		for (var i in onVolumeChangeHandlers)
			onVolumeChangeHandlers[i](self.ui)
	}

	this.onMuteChange = function(page, state) {
		var value = 0;
		if (page.get("value") != null) {
			var value = getTypedValue(page.get("value"))
			value = (value === true)
		}
		self.ui.mute = value
		for (var i in onMuteChangeHandlers)
			onMuteChangeHandlers[i](self.ui)
	}

	this.onDataChange = function(page, state) {
		var value = page.get("value") || {};

		/* Store currently player media */
		self.currentTrackRoles = value.trackRoles || null
		self.currentMediaRoles = value.mediaRoles || null

		/* Process player state to semi-fixed form, which can be easily used to
		 * update player screen
		 */

		/* fill in necessary structures */
		if (value.trackRoles == undefined) value.trackRoles = {}
		if (value.trackRoles.mediaData == undefined) value.trackRoles.mediaData = {}
		if (value.trackRoles.mediaData.metaData == undefined) value.trackRoles.mediaData.metaData = {}
		if (value.trackRoles.mediaData.resources == undefined) value.trackRoles.mediaData.resources = [{}]

		if (value.mediaRoles == undefined) value.mediaRoles = {}
		if (value.mediaRoles.mediaData == undefined) value.mediaRoles.mediaData = {}
		if (value.mediaRoles.mediaData.metaData == undefined) value.mediaRoles.mediaData.metaData = {}
		if (value.mediaRoles.mediaData.resources == undefined || value.mediaRoles.mediaData.resources.length == 0) value.mediaRoles.mediaData.resources = [{}]
		if (value.status == undefined) value.status = {}
		if (value.controls == undefined) value.controls = {}
		if (value.controls.playMode == undefined) value.controls.playMode = {}

		/* Parse contents into page variables */
		self.ui.title = value.trackRoles.title || value.mediaRoles.title || "";
		self.ui.artist = value.trackRoles.mediaData.metaData.artist || value.mediaRoles.mediaData.metaData.artist || value.trackRoles.description || "";
		self.ui.album = value.trackRoles.mediaData.metaData.album || value.mediaRoles.mediaData.metaData.album || "";
		self.ui.codec = value.trackRoles.mediaData.resources[0].codec || value.mediaRoles.mediaData.resources[0].codec || "";

		var bitrate = value.trackRoles.mediaData.resources[0].bitRate || value.mediaRoles.mediaData.resources[0].bitRate;
		var nominalBitrate = value.trackRoles.mediaData.resources[0].nominalBitRate || value.mediaRoles.mediaData.resources[0].nominalBitRate;
		var sampleRate = value.trackRoles.mediaData.resources[0].sampleFrequency || value.mediaRoles.mediaData.resources[0].sampleFrequency;
		var bps = value.trackRoles.mediaData.resources[0].bitsPerSample || value.mediaRoles.mediaData.resources[0].bitsPerSample;
		var channels = value.trackRoles.mediaData.resources[0].nrAudioChannels || value.mediaRoles.mediaData.resources[0].nrAudioChannels;
		var mqaInfo = value.trackRoles.mediaData.resources[0].mqaInfo || value.mediaRoles.mediaData.resources[0].mqaInfo;
		if (typeof mqaInfo == "undefined") {
			var mqaState = "";
		} else {
			var mqaState = mqaInfo.fullStringMqaInfo;
		}

		if (bitrate) bitrate = parseFloat(bitrate / 1000).toFixed(2) + " kbps";
		if (nominalBitrate) nominalBitrate = "(nominal: " + parseFloat(nominalBitrate / 1000).toFixed(2) + " kbps)";
		if (sampleRate) sampleRate = (sampleRate / 1000) + " kHz";
		if (bps) bps += " bits";
		if (channels) channels += " ch";

		/* Concatenate the strings, if they are not null and separate them with a comma */
		self.ui.audioFormat = $.grep([bitrate, nominalBitrate, sampleRate, bps, channels], Boolean).join(", ") || "";
		self.ui.mqa = "";
		if (mqaState) {
			self.ui.mqa = mqaState.replace(/^MQA Studio /, "<IMG src='img/mqa-studio.png' style='height:1.5em'><BR>");
		}

		self.ui.icon = value.trackRoles.icon || value.mediaRoles.icon || null;
		self.ui.playTimeTotal = value.status.duration || -1;

		self.ui.playId = value.playId || null;
		self.ui.contextPath = value.contextPath || null;
		self.ui.state = value.state || "stopped";
		self.ui.error = value.error || null;

		self.ui.buttons.next = value.controls.next_ || false;
		self.ui.buttons.previous = value.controls.previous || false;
		self.ui.buttons.pause = value.controls.pause || false;

		self.ui.seekTime = ('seekTime' in value.controls ? !value.controls.seekTime : false);

		self.ui.playmode.playModeSupported["normal"] = true;
		self.ui.playmode.playModeSupported["shuffle"] = value.controls.playMode.shuffle || false;
		self.ui.playmode.playModeSupported["repeatOne"] = value.controls.playMode.repeatOne || false;
		self.ui.playmode.playModeSupported["shuffleRepeatOne"] = value.controls.playMode.shuffleRepeatOne || false;
		self.ui.playmode.playModeSupported["repeatAll"] = value.controls.playMode.repeatAll || false;
		self.ui.playmode.playModeSupported["shuffleRepeatAll"] = value.controls.playMode.shuffleRepeatAll || false;

		if (self.ui.playmode.playModeSupported["shuffle"]) {
			self.ui.playmode.shuffle = true;
		} else {
			self.ui.playmode.shuffle = false;
		}

		if (self.ui.playmode.playModeSupported["repeatOne"] || self.ui.playmode.playModeSupported["repeatAll"]) {
			self.ui.playmode.repeat = true;
		} else {
			self.ui.playmode.repeat = false;
		}

		if (self.ui.icon == "skin:imageNoAlbumart")
			self.ui.icon = ""

		for (var i in onDataChangeHandlers)
			onDataChangeHandlers[i](self.ui)
	}

	this.control = function(action, args) {
		/* Valid actions according to NsdkPlayLogicControls:
		 * - pause
		 * - next_
		 * - previous
		 * - like
		 * - dislike
		 *
		 * Other functions as seekTime and seekTrack are not yet implemented.
		 */
		var ret = jQuery.Deferred();
		var value = {
			"control": action
		};
		if (action === "seekTime") {
			value.time = args;
		}

		NSDK.setData(controlPath, 'activate', value)
			.done(function() {
				;
				ret.resolve();
			})
			.fail(function(error) {
				ret.reject(error);
			});

		return ret.promise();
	}

	this.controlPlayMode = function(action) {
		var repeat = self.ui.buttons.repeat;
		var shuffle = self.ui.buttons.shuffle;

		if (action == "repeat") {
			for (var i = 0; i < 3; ++i) {
				repeat = (repeat + 1) % 3;
				if (checkPlaymodeSupported(shuffle, repeat, self.ui.playmode.playModeSupported, playModeStrings)) {
					break;
				}
			}
		} else if (action == "shuffle") {
			for (var i = 0; i < 2; ++i) {
				shuffle = (shuffle + 1) % 2;
				if (checkPlaymodeSupported(shuffle, repeat, self.ui.playmode.playModeSupported, playModeStrings)) {
					break;
				}
			}
		}

		var playModeCode = repeat * 2 + shuffle;
		var value = {
			'playerPlayMode': playModeStrings[playModeCode],
			'type': 'playerPlayMode',
		};

		return NSDK.setData(controlPlayModePath, 'value', value);
	}

	this.playContainer = function(containerRoles, trackRoles, index, version) {
		var data = {
			'control': 'play',
			'type': 'itemInContainer',
			'mediaRoles': containerRoles,
			'trackRoles': trackRoles,
			'index': index,
			'version': version,
		};
		return NSDK.setData(controlPath, 'activate', data);
	}

	this.play = function(mediaRoles) {
		var ret = $.Deferred();

		var data = {
			'control': 'play',
			'mediaRoles': mediaRoles,
		}
		NSDK.setData(controlPath, 'activate', data)
			.done(function() {
				ret.resolve();
			})
			.fail(function(error) {
				var result = error.error.actionReply
				if (!$.isEmptyObject(result) && !$.isEmptyObject(result.xclass) && result.xclass == "NsdkActionReply") {
					// close all necessary paths
					if (result.events) {
						for (var i = 0; i < result.events.length; i++) {
							var event = result.events[i];
							console.log(event.path + ' ' + event.rowsType);
							if (event.rowsType == "remove")
								browser.closePage(event.path);
						}
					}
					// redirect if a redirect path is set */
					if (result.result && result.result.path) {
						browser.openPage(result.result.path, false);
						ret.resolve(result.result.path);
					} else if (result.result && result.result.type == "www") {
						window.open(result.result.mediaData.resources[0].uri, "_blank")
					}
					// refresh current page
					else {
						browser.deactivateCurrent();
						browser.activateCurrent();
					}
				} else {
					ret.resolve(result);
				}
			});

		return ret.promise()
	}

	this.seek = function(time) {
		self.control({
			'control': 'seek',
			'time': time
		})
	};

	this.setVolume = function(value) {
		if (data_volumePath != null)
			NSDK.setData(data_volumePath, 'value', value);
	}

	this.setMute = function(value) {
		if (data_mutePath != null)
			NSDK.setData(data_mutePath, 'value', value);
	}

	$.when(
		NSDK_PROXY.browse("player:player/control").done(function(a) {
			controlPath = a;
		}),
		NSDK_PROXY.browse("player:player/data").done(function(a) {
			dataPath = a;
		}),
		NSDK_PROXY.browse("settings:/mediaPlayer/playMode").done(function(a) {
			controlPlayModePath = a;
		}),
		NSDK_PROXY.browse("player:player/data/playTime").done(function(a) {
			data_playTimePath = a;
		}),
		NSDK_PROXY.browse("player:volume").done(function(a) {
			data_volumePath = a;
		}),
		NSDK_PROXY.browse("settings:/mediaPlayer/mute").done(function(a) {
			data_mutePath = a;
		})
	).done(function() {
		NSDK_PROXY.subscribe(controlPlayModePath, 'update', self.onPlayModeChange);
		NSDK_PROXY.subscribe(data_playTimePath, 'update', self.onPlayTimeChange);
		NSDK_PROXY.subscribe(data_volumePath, 'update', self.onVolumeChange);
		NSDK_PROXY.subscribe(data_mutePath, 'update', self.onMuteChange);
		NSDK_PROXY.subscribe(dataPath, 'update', self.onDataChange);
	}).fail(function() {
		console.log("Failed to initialize PlayerController");
	})
}
