#!/bin/bash
# common parts of script for generating signed swupdate file

set -e

source "$(dirname $(readlink -f "$0"))/sign_swupdate_common.config"

# use parallel gzip if available
[ -x "$(command -v pigz)" ] && GZIP=pigz || GZIP=gzip


create_dtb_file_pubkey() {
	# we need to create a minimalistic device-tree file where mkimage will store the
	# public key in the right format (to be appended to u-boot binary then)
	DTB_FILE_PUBKEY="${DEPLOY_DIR_IMAGE}/fitImage-pubkey-${CONFIG_MACHINE}-${DATETIME}"

	if [ -e "${DTB_FILE_PUBKEY}.dtb" ]; then
		echo "==== DTB for the public key already exists, reusing..."
	else
		echo "==== Creating DTB for the public key..."
		cat << EOF > "${DTB_FILE_PUBKEY}".dts
/dts-v1/;
/ {};
EOF

		dtc -p 0x1000 "${DTB_FILE_PUBKEY}.dts" -O dtb -o "${DTB_FILE_PUBKEY}.dtb"
	fi
}

inject_wrap_key_hmac_into_swupdate() {
	local TMP_ROOTFS="$1"
	if [ ${#WRAP_KEY} -eq 64 ] && [ ${#WRAP_HMAC} -eq 32 ] && [ -f "${TMP_ROOTFS}/usr/bin/swupdate" ]; then
		OFFSET=$(grep --byte-offset --only-matching --text -E 'a{16}b{32}' "${TMP_ROOTFS}/usr/bin/swupdate" | sed 's/:.*//')

		dd if="${TMP_ROOTFS}/usr/bin/swupdate" count=1 bs=${OFFSET} > "${TMP_ROOTFS}/usr/bin/swupdate_tmp"
		echo -n "${WRAP_HMAC}${WRAP_KEY}" | xxd -r -p >> "${TMP_ROOTFS}/usr/bin/swupdate_tmp"
		dd if="${TMP_ROOTFS}/usr/bin/swupdate" bs=$((${OFFSET} + 48))  skip=1 >> "${TMP_ROOTFS}/usr/bin/swupdate_tmp"
		mv "${TMP_ROOTFS}/usr/bin/swupdate_tmp" "${TMP_ROOTFS}/usr/bin/swupdate"
		chmod +x "${TMP_ROOTFS}/usr/bin/swupdate"
	fi
}

insert_keys_into_swu_rootfs() {
	echo "== Packing image verification key into SWU image..."

	local TMP_SWUPDATE_ROOTFS="${TMP_DIR}/swu_rootfs"
	mkdir -p "${TMP_SWUPDATE_ROOTFS}"

	echo "==== Extracting swupdate rootfs to a temporary directory..."
	zcat "${DEPLOY_DIR_IMAGE}/swupdate-image-${CONFIG_MACHINE}.cpio.gz" | (cd "${TMP_SWUPDATE_ROOTFS}" && cpio -id)

	echo "==== Inserting verification key into the swupdate rootfs..."
	openssl rsa -in "${DEPLOY_DIR_IMAGE}/boot_keys/verified-boot.key" -pubout -outform PEM -out "${TMP_SWUPDATE_ROOTFS}/home/root/verified-boot.pem"

	inject_wrap_key_hmac_into_swupdate "${TMP_SWUPDATE_ROOTFS}"

	echo "==== Packing the swupdate rootfs back together..."
	(cd "${TMP_SWUPDATE_ROOTFS}" && find . | cpio -R 'root:' -o -H newc) | ${GZIP} -f -9 > "${DEPLOY_DIR_IMAGE}/swupdate-image-${CONFIG_MACHINE}-${DATETIME}-keys.rootfs.cpio.gz"
	ln -sf "swupdate-image-${CONFIG_MACHINE}-${DATETIME}-keys.rootfs.cpio.gz" "${DEPLOY_DIR_IMAGE}/swupdate-image-${CONFIG_MACHINE}-keys.cpio.gz"
}

insert_various_items_into_rootfs()
{
	echo "== Packing signed TAs and fit public key into rootfs image..."

	# Differentiate between ubifs and ext4
	if grep "\-${CONFIG_MACHINE}.ubifs" "${DEPLOY_DIR_IMAGE}/sw-description"; then
		ROOTFS_TYPE="ubifs"
	elif grep "\-${CONFIG_MACHINE}.ext4" "${DEPLOY_DIR_IMAGE}/sw-description"; then
		ROOTFS_TYPE="ext4"
	else
		echo "Error: Unsupported rootfs type!"
		exit 1
	fi
	#           grep for all lines containing valid filename entry                 | extract only the filename    | grep only for rootfs       | extract only imagename
	IMAGENAME=(`grep -o "filename[ ]*=[^;]*;" "${DEPLOY_DIR_IMAGE}/sw-description" | sed 's:.*"\(.*\)"[^"]*:\1:g' | grep "\-${CONFIG_MACHINE}.${ROOTFS_TYPE}$" | sed "s:-${CONFIG_MACHINE}.${ROOTFS_TYPE}\$::g"`)
	TALIST=""
	for N in ${TMP_DIR}/ta/*.ta; do
		[ -e "$N" ] || continue
		BN=`basename "$N"`
		TALIST="$TALIST $BN"
	done

	echo "==== Extracting rootfs tarball..."
	local TMP_ROOTFS="${TMP_DIR}/rootfs"
	mkdir -p "${TMP_ROOTFS}"
	bzip2 -cd "${DEPLOY_DIR_IMAGE}/${IMAGENAME}-${CONFIG_MACHINE}.tar.bz2" | /usr/bin/fakeroot tar xf - -C "${TMP_ROOTFS}"

	if [ -n "$TALIST" ]; then
		# make sure the destination dir exists
		mkdir -p "${TMP_ROOTFS}/lib/optee_armtz"

		for BN in $TALIST; do
			echo "==== Installing $BN..."
			cp "${TMP_DIR}/ta/$BN" "${TMP_ROOTFS}/lib/optee_armtz/$BN"
		done
	else
		echo "==== No TAs found"
	fi

	# make sure the destination dir exists
	mkdir -p "${TMP_ROOTFS}/home/root"
	echo "==== Inserting verification keys into rootfs..."
	cp "${DEPLOY_DIR_IMAGE}/fitImage-pubkey-${CONFIG_MACHINE}-${DATETIME}.dtb" "${TMP_ROOTFS}/home/root/verified-boot.dtb"
	openssl rsa -in "${DEPLOY_DIR_IMAGE}/boot_keys/verified-boot.key" -pubout -outform PEM -out "${TMP_ROOTFS}/home/root/verified-boot.pem"

	inject_wrap_key_hmac_into_swupdate "${TMP_ROOTFS}"

	insert_platform_specific_items_into_rootfs "$TMP_ROOTFS"

	if [ "$ROOTFS_TYPE" = "ubifs" ]; then
		OUT="${DEPLOY_DIR_IMAGE}/${IMAGENAME}-${CONFIG_MACHINE}-${DATETIME}.rootfs.ubifs"
		MKUBIFS_ARGS=`cat "${DEPLOY_DIR_IMAGE}/signing_scripts/params.mkubifs"`
		echo "==== Re-creating ubifs image..."
		/usr/bin/fakeroot "${DEPLOY_DIR_IMAGE}/signing_scripts/mkfs.ubifs" ${MKUBIFS_ARGS} -r "${TMP_ROOTFS}" -o "$OUT"
		ln -sf "${IMAGENAME}-${CONFIG_MACHINE}-${DATETIME}.rootfs.ubifs" "${DEPLOY_DIR_IMAGE}/${IMAGENAME}-${CONFIG_MACHINE}.ubifs"
	else
		OUT="${DEPLOY_DIR_IMAGE}/${IMAGENAME}-${CONFIG_MACHINE}-${DATETIME}.rootfs.ext4"
		local IMG_OVERHEAD_FACTOR="13/10" # 1.3 the same in yocto
		local ACTUAL_ROTFS_SIZE=$(du -s ${TMP_ROOTFS} | awk '{print $1}')
		local ROOTFS_SIZE=$(($ACTUAL_ROTFS_SIZE * $IMG_OVERHEAD_FACTOR ))

		dd if=/dev/zero of=${OUT} seek=${ROOTFS_SIZE} count=0 bs=1024
		extra_imagecmd="-i 4096" #from yocto
		/usr/bin/fakeroot "mkfs.ext4" -F $extra_imagecmd ${OUT} -d "${TMP_ROOTFS}"
		# Error codes 0-3 indicate successfull operation of fsck (no errors or errors corrected
		fsck.ext4 -pvfD ${OUT} || [ $? -le 3 ]

		ln -sf "${IMAGENAME}-${CONFIG_MACHINE}-${DATETIME}.rootfs.ext4" "${DEPLOY_DIR_IMAGE}/${IMAGENAME}-${CONFIG_MACHINE}.ext4"
	fi

}

create_swupdate_file() {
	# expecting:
	#	DEPLOY_DIR_IMAGE/sw-description
	#	DEPLOY_DIR_IMAGE/boot_keys/verified-boot.key
	# output:
	#	DEPLOY_DIR_IMAGE/image-${CONFIG_MACHINE}-${DATETIME}-signed.swu
	#	DEPLOY_DIR_IMAGE/image.swu (link)

	[ "$ENCRYPTION" == "true" ] && ENC="-enc" || ENC=""
	echo "== Creating swupdate${ENC}${COMP} file..."

	# Parse file names from sw-description
	DESC_FILES=(`grep -o "filename[ ]*=[^;]*;" "${DEPLOY_DIR_IMAGE}/sw-description" | sed 's:.*"\(.*\)"[^"]*:\1:g'`)

	cp "${DEPLOY_DIR_IMAGE}/sw-description" "${TMP_DIR}/sw-description"

	echo "==== Copying files to a temporary directory..."
	for FILE in ${DESC_FILES[@]}; do
		FILE_PATH="${DEPLOY_DIR_IMAGE}/${FILE}"
		if [ "${FILE: -6}" = ".ubifs" -o "${FILE: -5}" = ".ext4" ]; then
			# NOTE: only use xz compression level "-6": this requires 9MB of ram for decompression
			#       (-9 requires 65MB which is too much for us)
			xz -T0 -6 -c "$FILE_PATH" > "${TMP_DIR}/${FILE}.xz"
			FILE_PATH="${TMP_DIR}/${FILE}.xz"
		fi
		if [ "$ENCRYPTION" == "true" ] && [ "${FILE: -3}" != ".sh" ]; then
			openssl enc -aes-256-cbc -in "$FILE_PATH" -out "${TMP_DIR}/${FILE}" -K "${AES_KEY}" -iv "${AES_IV}" -nosalt
		else
			cp "$FILE_PATH" "${TMP_DIR}/${FILE}"
		fi
	done

	# Replace @filename with properties of the file (sha256, fileSize, sha256 of decrypted file, encryption enabled)
	echo "==== Calculating SHA-256 hashes for the sw-description file..."
	for FILE in ${DESC_FILES[@]}; do
		REPLACE="sha256 = \"$(sha256sum ${TMP_DIR}/$FILE | sed 's: .*::g')\";\n\
			filesize = \"$(stat -L -c%s ${DEPLOY_DIR_IMAGE}/$FILE)\";"
		if [ "${FILE: -6}" = ".ubifs" -o "${FILE: -5}" = ".ext4" ]; then
			REPLACE="$REPLACE\n\
			compressed = true;"
		fi
		if [ "$ENCRYPTION" == "true" ] && [ "${FILE: -3}" != ".sh" ]; then
			REPLACE="$REPLACE\n\
			sha256dec = \"$(sha256sum ${DEPLOY_DIR_IMAGE}/$FILE | sed 's: .*::g')\";\n\
			encrypted = true;\
			"
		fi
		sed -i "s:@$FILE:$REPLACE:g" "${TMP_DIR}/sw-description"
	done

	# Update uboot and fit versions
	VERIFY_KEY_HASH=$(sha256sum "${DEPLOY_DIR_IMAGE}/boot_keys/verified-boot.key" | cut -d ' ' -f 1)
	sed -i "s:@VERKEY:x${VERIFY_KEY_HASH:0:7}:g" ${TMP_DIR}/sw-description

	echo "==== Signing sw-description..."
	openssl dgst -sha256 -sign "${DEPLOY_DIR_IMAGE}/boot_keys/verified-boot.key" "${TMP_DIR}/sw-description" > "${TMP_DIR}/sw-description.sig"

	echo "==== Packing files into the final image..."
	POSTFIX="${ENC}.swu"

	# sw-descriptions have to come first
	(cd ${TMP_DIR} && echo "sw-description sw-description.sig ${DESC_FILES[@]}" | tr ' ' '\n' | cpio -o -H crc) > "${DEPLOY_DIR_IMAGE}/image-${CONFIG_MACHINE}-${DATETIME}-signed${POSTFIX}"
	ln -sf "image-${CONFIG_MACHINE}-${DATETIME}-signed${POSTFIX}" "${DEPLOY_DIR_IMAGE}/image${POSTFIX}"

	echo "== Creating swupdate file complete!"
}

create_configured_swupdate_files() {
	# always create un-encrypted image
	ENCRYPTION=false create_swupdate_file

	# if encryption keys are available, also create encrypted image
	if [ "${CONFIG_USE_ENCRYPTION}" = true ]; then
		ENCRYPTION="${CONFIG_USE_ENCRYPTION}" create_swupdate_file
	fi
}

# beginning of the script

# global variables
DEPLOY_DIR_IMAGE="$1"
# make absolute path from it - we need to cd/pushd for mkimage run so make sure we use proper paths
# (realpath is not installed by default on ubuntu so just check for leading /)
[ "${DEPLOY_DIR_IMAGE:0:1}" == "/" ] || DEPLOY_DIR_IMAGE="`pwd`/$DEPLOY_DIR_IMAGE"


if [ $# -eq 3 ]; then
	DATETIME="$2"
	MODE="$3"
elif [ $# -eq 4 ]; then
	if [ "$2" != "$CONFIG_MACHINE" ]; then
		echo "Error: specified MACHINE does not match the config ($2 != $CONFIG_MACHINE)!"
		exit 1
	fi
	DATETIME="$3"
	MODE="$4"
else
	echo "Error: invalid number of parameters!"
	usage
	exit 1
fi


# optional image encryption support
if [ "${CONFIG_USE_ENCRYPTION}" = true ]; then
	echo "== Encryption enabled"

	if [ ! -f "${DEPLOY_DIR_IMAGE}/swu_keys/enc_key" ]; then
		echo "Missing enc_key"
		exit 1
	fi

	if [ ! -f "${DEPLOY_DIR_IMAGE}/swu_keys/wrap_key" ]; then
		echo "Missing wrap_key"
		exit 1
	fi

	WRAP_KEY=$(grep "^key *= *" "${DEPLOY_DIR_IMAGE}/swu_keys/wrap_key" | sed 's/^key *= *//')
	WRAP_HMAC=$(grep "^hmac *= *" "${DEPLOY_DIR_IMAGE}/swu_keys/wrap_key" | sed 's/^hmac *= *//')
	if [ ${#WRAP_KEY} -ne 64 ] || [ ${#WRAP_HMAC} -ne 32 ]; then
		echo "Invalid wrap_key format"
		exit 1
	fi

	AES_KEY=$(grep "^key *= *" "${DEPLOY_DIR_IMAGE}/swu_keys/enc_key" | sed 's/^key *= *//')
	AES_IV=$(grep "^iv *= *" "${DEPLOY_DIR_IMAGE}/swu_keys/enc_key" | sed 's/^iv *= *//')
	if [ -z "${AES_KEY}" ] || [ -z "${AES_IV}" ]; then
		echo "Invalid aes key file format"
		exit 1
	fi
fi


# force paths for mkimage to local libs (ssl, crypto)
YOCTO_LIBRARY_PATH="${DEPLOY_DIR_IMAGE}/signing_scripts"


# create temp dir to be used during the image generation
TMP_DIR=`mktemp -d -t`
trap "rm -rf $TMP_DIR; exit" SIGHUP SIGINT SIGTERM EXIT
