"use strict";

var popupWin;
var oauthInvokingBrowser;

// function called after succesful oauth login
window.addEventListener("message", function(event) {
	if (event.data == "oauth-success" && oauthInvokingBrowser) {
		oauthInvokingBrowser.openPage(oauthInvokingBrowser.page_stack[oauthInvokingBrowser.page_stack.length - 1].getRequestedPath(), {});
		window.setTimeout(function() {
			popupWin.close();
		}, 5000);
		window.focus();
	}
});

function BrowserItem(my_id, row_data, my_version, my_page) {
	var self = this;
	var id = my_id;
	var row = row_data;
	var page = my_page;
	var editor = null;
	var version = my_version;

	this.getPlayerController = function() {
		return page ? page.getPlayerController() : null;
	};

	this.getBrowser = function() {
		return page ? page.getBrowser() : null;
	};

	this.destroy = function() {
		editor = null;
		// Break cycle references between BrowserPage and BrowserItem
		page = null;
	}

	/**
	 * Renders one row of the listing
	 * @param row roles associated with currently-rendered row
	 * @return string containining HTML code for 'row'
	 */
	this.render = function() {
		var content = {};

		if (row == null || row == undefined || !page)
			return content;

		content.class = self.getClassName();
		content.html = self.getContent();

		return content;
	}

	this.getContent = function() {
		var isEnumSubmenu = false;
		var icon = "";
		var content = "";

		if (row == null || row == undefined || !page)
			return content;

		if (row.icon != null) {
			/* icon is not null */
			icon = iconPath(row.icon);
		}

		/* check box and radio icons are not propagated if not set directly to a node
		 handle them separately */
		if (row.type == "value" && row.value.type == "bool_") {
			if (getTypedValue(row.value))
				icon = iconPath("skin:iconCheckboxChecked");
			else
				icon = iconPath("skin:iconCheckboxUnchecked");
		}

		if (row.edit && row.edit.type == "enum_")
			icon = iconPath("skin:iconEnum");

		if (row.type != null && icon == "") {
			/* type is not null */
			switch (row.type) {
				case 'container':
					icon = iconPath("skin:iconContainer");
					break;
				case 'audio':
					icon = iconPath("skin:iconAudio");
					break;
			}
		}

		var context = page.getContext();
		if (!self.isDummy() && context.relatedRoles && context.relatedRoles.edit && context.relatedRoles.edit.type == "enum_") {
			isEnumSubmenu = true;
			icon = iconPath("skin:iconCheckboxUnchecked");
			if (context.relatedRoles.value[context.relatedRoles.value.type] == row.value[row.value.type])
				icon = iconPath("skin:iconCheckboxChecked");
		}

		if (icon == "")
			icon = iconPath("skin:iconBlank");

		content += '<div class="item-icon"><img src="' + textToHtml(icon) + '"/></div>';

		// Render row label
		content += '<div class="item-name">'

		switch (row.type) {
			case "spacer":
				content += '<span class="delimiter">~~~~~~~~~~~~~~~~~~~</span>';
				break;
			case "footer":
			case "header":
				if (row.title)
					content += '<span class="delimiter">~~~ </span><span class="title">' + textToHtml(row.title) + '</span><span class="delimiter"> ~~~</span>'; // ~~~ title ~~~
				else
					content += '<span class="delimiter">~~~~~~~</span>';
				break;
			default:
				if (row.title) {
					content += '<span class="title">' + textToHtml(row.title) + "</span>";
					// Append badges after the title
					if (row.mediaData && row.mediaData.metaData && row.mediaData.metaData.comment) {
						var cssStyle = "display: inline; padding: .2em .3em .3em; font-size: 75%; font-weight: 700; line-height: 1; color: #fff; text-align: center; white-space: nowrap; vertical-align: baseline; border-radius: .25em; background-color: #333";
						content += " <span style=\"" + cssStyle + "\">" + row.mediaData.metaData.comment + "</span>";
					}
					if (!($.isEmptyObject(row.description)))
						content += '<span class="delimiter"> | </span><span class="description">' + textToHtml(row.description) + "</span>";
				} else
					content += '<span class="title unknown">unknown</span>';
				break;
		}

		/* Handle 'value' type of rows' */
		if (row.type == "value" && row.value.type != "bool_") {
			// If the value has 'edit' property and its of type "enum_", do not append the value
			if ((row.edit && row.edit.type && row.edit.type == "enum_") || (isEnumSubmenu == true)) {
				console.log("Skip appending if an enum submenu");
			} else {
				content += '<span class="delimiter">: </span><span class="value">' + textToHtml(getTypedValue(row.value)) + "</span>";
				if (row.valueUnit) {
					content += '<span class="delimiter"> </span><span class="unit">' + textToHtml(getTypedValue(row.valueUnit)) + "</span>";
				};
			};
		};

		if (row.type == "query") {
			content += '<span class="value"></span>';
		}

		content += "</div>";

		if (row.context && row.context.path)
			content += '<div class="context"><img src="' + textToHtml(iconPath("skin:iconContext")) + '"/></div>';
		else
			content += '<div class="context none">&nbsp;</div>';

		return content;
	}


	/**
	 * Either opens a context menu (if ctx === true and context
	 * exists for this path) or performs an action.
	 */
	this.onClick = function(ctx) {
		if (row == null || row == undefined || !page)
			return;

		if (ctx && row.context && row.context.path) {
			self.getBrowser().openPage(row.context.path, {
				'relatedRoles': row
			});
		} else {
			self.action();
		}
	}

	/* Helper to execute setData request via NsdkProxy. This requires creation
	 * of temporary NsdkPath instance and then performing required request.
	 */
	this.setData = function(path, item, new_value) {
		var ret = $.Deferred();

		NSDK.setData(path, item, new_value, null)
			.done(function(result) {
				// save browser so all events are executed even if current page is closed in NsdkActionReply
				var browser = self.getBrowser();
				if (!$.isEmptyObject(result) && !$.isEmptyObject(result.xclass) && result.xclass == "NsdkActionReply") {
					// close all necessary paths
					if (result.events) {
						for (var i = 0; i < result.events.length; i++) {
							var event = result.events[i];
							console.log(event.path + ' ' + event.rowsType);
							if (event.rowsType == "remove") {
								// browser will be null if page was deleted duting the async call
								if (browser)
									browser.closePage(event.path);
							}
						}
					}
					// redirect if a redirect path is set
					if (result.result && result.result.path) {
						ret.resolve(result.result.path, browser);
					} else if (result.result && result.result.type == "www") {
						window.open(result.result.mediaData.resources[0].uri, "_blank")
					}
					// refresh current page
					else {
						// self.getBrowser() will be null if page was deleted duting the async call
						if (self.getBrowser()) {
							self.getBrowser().deactivateCurrent();
							self.getBrowser().activateCurrent();
						}
					}
				} else {
					ret.resolve(result, browser);
				}
			})
			.fail(function(error) {
				console.log(error)
				ret.reject(error)
			});

		return ret.promise()
	}

	/**
	 * Helper for setting path to specified value
	 */
	this.setDataValue = function(row, new_value) {
		return self.setData(row.path, 'value', stringToValue(new_value, row.value.type))
	}

	/**
	 * Handles clicking on particular item inside content browser.
	 * @param rowId index of an item inside its folder (extracted from
	 *              ID of its tag.
	 * @return True or false, depending on whether content browser has
	 *         to reload contents.
	 */
	this.action = function() {
		var type = row.type;
		/* Do not allow clicking on dummy items */
		if (self.isDummy()) {
			window.alert("Please wait until the item is loaded!");
			return;
		}

		/* Do not allow clicking on disabled items */
		if (row.disabled) {
			console.log("Item is disabled");
			return;
		}

		// Exit if destroy was already called
		if (!page || !self.getBrowser()) {
			return;
		}

		// Exit if destroy was already called
		if (!self.getBrowser()) {
			return;
		}

		switch (type) {
			case 'container':
				var res = {}
				$.extend(res, page.getContext(), {
					'roles': row
				});
				self.getBrowser().openPage(row.path, res);
				break;

			case 'audio':
				if (!self.getPlayerController()) {
					break;
				}
				if (row.audioType && (row.audioType == "audioBroadcast"))
					self.getPlayerController().play(
						row /* trackRoles */ );
				else
					self.getPlayerController().playContainer(
						page.getContext().roles, /* containerRoles */
						row /* trackRoles */ ,
						id,
						parseInt(version));
				break;
			case 'separator':
			case 'spacer':
			case 'header':
			case 'footer':
				break;
			case 'value':
				/* Check whether we are in nested menu, which is usually
				 * invoked when looking for a enum_ value. If so, just call
				 * supplied callback and close the menu. */
				if (page.getContext() && page.getContext()['value-click']) {
					page.getContext()['value-click'](row.value);
					return;
				}

				/* Check if the row is modifiable or not. If not just return */
				if (!row.hasOwnProperty('modifiable') || row.modifiable !== true) {
					console.log("Not modifiable row!");
					return;
				}

				var newval;
				if (row.edit && row.edit.type && row.edit.type == "enum_") {
					self.setupEditor();
				} else {
					switch (row.value.type) {
						case 'bool_':
							newval = (!getTypedValue(row.value)).toString();
							if (newval)
								self.setDataValue(row, newval).fail(window.alert);
							break;
						case 'double_':
						case 'i32_':
						case 'string_':
						case 'enum_':
							self.setupEditor();
							break;
						default:
							newval = window.prompt(row.title, getTypedValue(row.value));
							if (newval)
								self.setDataValue(row, newval).fail(window.alert);
					}
				}

				break;
			case 'action':
				self.getBrowser().onWorkingStateChange(true);
				self.setData(row.path, 'activate', true)
					.always(function() {
						//whatever the result of the request is
						//spinning div has to be gone when the reply
						//was received
						// self.getBrowser() will be null if page was deleted during the async call
						// in that case we dont care about WorkingStateChange anymore
						if (self.getBrowser())
							self.getBrowser().onWorkingStateChange(false);
					})
					.done(function(redirect, browser) {
						if (redirect) {
							var res = {}
							// page and self.getBrowser() will be null if page was deleted during the async call
							if (page)
								$.extend(res, page.getContext(), {
									'roles': row
								});

							if (browser) {
								browser.closeContext(redirect);
								browser.openPage(redirect, res);
							}
						}
					})
					.fail(function(data) {
						console.log("Setting value failed");
						console.log(data)
						window.alert(data.error.message);
					});

				break;
			case 'query':
				editor = new StringEditor(stringToValue("", "string_"),
					null,
					self.doQuery,
					function() {
						$(self.getBrowser().getTableViewSelector() + " .item[data-id='" + id + "'] .value").html("");
					}

				);
				$(self.getBrowser().getTableViewSelector() + " .item[data-id='" + id + "'] .value").html(editor.render());
				editor.bindIDs();
				editor.focus();
				break;
			case 'www':
				// if the url is not the a Oauth login one, open it normally
				window.open(row.mediaData.resources[0].uri, "_blank", "resizable=0,width=480,height=640");
				break;

			case 'wwwOauth':
				// if the webpage to load is the Amazon Login page
				oauthInvokingBrowser = self.getBrowser();
				oauthInvokingBrowser.onWorkingStateChange(true);

				// open a popUp window directed to the amazon login url and save the url for being able to close it after the login
				popupWin = window.open(row.mediaData.resources[0].uri, "_blank");
				break;

			default:
				console.log("This item is not playable (type '" + type + "')");
		}
	};

	this.doQuery = function(str) {
		self.setData(row.path, 'query', stringToValue(str, "string_"))
			.done(function(param_data, browser) {
				// self.getBrowser() will be null if page was deleted duting the async call
				if (browser)
					browser.openPage(param_data, page.getContext());
			})
			.fail(window.alert)
		$(self.getBrowser().getTableViewSelector() + " .item[data-id='" + id + "'] .value").html("");
	}

	this.showValue = function() {
		if (!editor)
			return;
		$(self.getBrowser().getTableViewSelector() + " .item[data-id='" + id + "'] .value").html(textToHtml(getTypedValue(row.value).toString()));
		editor = null;
	}

	this.setupEditor = function() {
		/* string editor */
		if (editor)
			return;

		if ($.isEmptyObject(row.edit) == false && row.edit.type) {
			switch (row.edit.type) {
				case "enum_":
					var context = {};
					$.extend(true, context, page.getContext(), {
						'value-click': function(value) {
							var pathToClose = row.path;
							self.setData(row.path, 'value', value)
								.done(function(data, browser) {
									browser.closePage(pathToClose);
								})
						},
						'relatedRoles': row
					});
					self.getBrowser().openPage(row.edit.enumPath, context);
					editor = null;
					break;
				case "slider":
					editor = new NumberEditor(row.value,
						parseFloat(row.edit.min),
						parseFloat(row.edit.max),
						parseFloat(row.edit.step),
						self.updateValue,
						self.saveValue,
						self.showValue
					);
					break;
				case "none":
					editor = new StringEditor(row.value,
						self.updateValue,
						self.saveValue,
						self.showValue);
					break;

				default:
					window.alert("Unknown edit type: " + row.edit.type)
					break;
			}
		} else {
			switch (row.value.type) {
				case 'string_':
					editor = new StringEditor(row.value,
						self.updateValue,
						self.saveValue,
						self.showValue
					);
					break;
				case 'double_':
				case 'i32_':
					editor = new NumberEditor(row.value,
						parseFloat(row.edit.min),
						parseFloat(row.edit.max),
						parseFloat(row.edit.step),
						self.updateValue,
						self.saveValue,
						self.showValue
					);
					break;

			}
		}
		if (editor) {
			$(self.getBrowser().getTableViewSelector() + " .item[data-id='" + id + "'] .value").html(editor.render());
			editor.bindIDs();
			editor.focus();
		}
	}

	this.updateValue = function(val) {
		var new_value = val;
		/* if the value has changed, save it*/
		if (getTypedValue(row.value).toString() != new_value) {
			// TODO: replace by polling the proper key
			row.value = stringToValue(new_value, row.value.type);
			return self.setDataValue(row, new_value)
			//return self.getBrowser().api.setData (row.path, 'value', stringToValue (new_value, row.value.type), null).error(window.alert);

		};
	}

	this.saveValue = function(val) {
		var ret = self.updateValue(val)
		self.showValue();
	}

	/**
	 * Get the correct class name for this item
	 * @return Class name as a string
	 */
	this.getClassName = function() {
		if (self.isDummy() || row.disabled)
			return "item item-type-disabled";
		else
			return "item item-type-" + row.type;
	}

	/**
	 * Tell if the row is a dummy row or has content
	 * @return boolean indicating if it is a dummy or not
	 */
	this.isDummy = function() {
		if (row.type == "dummy")
			return true;
		else
			return false;
	}
}

/*
 * Various editors
 */

function StringEditor(value, change, ok, cancel) {
	var self = this;
	var id = generateRandomElementId();
	var value = value;
	var change = change;
	var ok = ok;
	var cancel = cancel;

	this.render = function() {
		var ret = '<br><input id="' + id + '-value-editor" class="value-editor" type="text" value="' +
			textToHtml(getTypedValue(value).toString()) + '"></input>';
		ret += '<input id="' + id + '-ok" class="ok-button" type="button" value="OK"/>';
		ret += '<input id="' + id + '-cancel" class="cancel-button" type="button" value="X"/>';
		return ret;
	}

	this.getValue = function() {
		return $("#" + id + "-value-editor").prop("value");
	}

	this.bindIDs = function() {
		$("#" + id + "-value-editor").click(function(event) {
			event.stopPropagation();
		});
		$("#" + id + "-value-editor").focusout(function(event) {
			ok(self.getValue());
			event.stopPropagation();
		});
		$("#" + id + "-ok").click(function(event) {
			ok(self.getValue());
			event.stopPropagation();
		});
		$("#" + id + "-cancel").mousedown(function(event) {
			if (cancel) {
				cancel();
			}
			event.stopPropagation();
		});
	}

	this.focus = function() {
		$("#" + id + "-value-editor").focus();
	}
}

function NumberEditor(value, nmin, nmax, nstep, change, ok, cancel) {
	var self = this;
	var id = generateRandomElementId();
	var value = value;
	var change = change;
	var ok = ok;
	var cancel = cancel;

	var nmin = nmin;
	var nmax = nmax;
	var nstep = nstep;

	this.render = function() {
		var ret;
		ret = '<input id="' + id + '-less" type="button" value="&lt;"></input>&nbsp;';
		ret += '<div id="' + id + '-container" class="range-container"><span id="' + id + '-editor">' + getTypedValue(value) + '</span><div id="' + id + '-range" class="range"></div></div>&nbsp;'
		ret += '<input id="' + id + '-more" type="button" value="&gt;"></input>';
		ret += '<input id="' + id + '-cancel" type="button" value="X"></input>';
		return ret;
	}

	this.getValue = function() {
		return parseFloat($("#" + id + "-editor").html())
	}

	this.bindIDs = function() {
		$("#" + id + "-less").click(
			function() {
				self.setValue(Math.max(nmin, self.getValue() - nstep));
			});
		$("#" + id + "-more").click(
			function() {
				self.setValue(Math.min(nmax, self.getValue() + nstep));
			});
		$("#" + id + "-cancel").click(
			function(event) {
				cancel();
				event.stopPropagation();
			});
	}

	this.humanizeValue = function(value) {
		//the call to replace removes trailing zeroes
		return value.toPrecision(9).replace(/\.?0*$/, '');
	}

	this.setValue = function(value) {
		$("#" + id + "-editor").html(self.humanizeValue(value));
		$("#" + id + "-range").css("right", (100 - (value - nmin) / (nmax - nmin) * 100).toString() + "%");
		self.valueUpdated();

	}

	this.valueUpdated = function() {
		change(self.getValue())
	}

	this.focus = function() {
		$("#" + id + "-editor").focus();
		self.setValue(self.getValue());
	}
}
