"use strict";

function EventHandler() {
	this.pending_updates = [];
	this.promise;
	this.subscriptions = {};
	this.browsers = [];
}

EventHandler.prototype.handle = function(event) {
	var self = this;
	var ret;
	console.log("handle: " + JSON.stringify(event));
	if (event.hasOwnProperty("rowsType")) {
		if (event.rowsType == "update" && event.hasOwnProperty("rowsEvents")) {
			ret = this.handleRowsUpdateEvent(event);
		} else if (event.rowsType == "reset") {
			ret = this.handleRowsResetEvent(event);
		} else {
			ret = this.handleRowsRemoveEvent(event);
		}
		$.when(ret).done(function() {
			self.browsers.forEach(function(browser) {
				if (browser.page_stack.length == 0) {
					return;
				}
				var page = browser.page_stack[browser.page_stack.length - 1];
				if (page.getPath() == event.path) {
					console.log("Updating " + event.path + "...");
					page.renderContentTable();
				}
			});
		});
	} else if (event.hasOwnProperty("itemType")) {
		if (event.itemType == "update") {
			if (this.subscriptions[event.path]) {
				NSDK_PROXY.pageAt(event.path).refresh()
					.done(function() {
						self.notify(event.path, event.itemType);
					});
			}
		}
	}
};

EventHandler.prototype.notify = function(eventPath, eventType) {
	if (this.subscriptions[eventPath] && this.subscriptions[eventPath][eventType]) {
		this.subscriptions[eventPath][eventType](NSDK_PROXY.pageAt(eventPath));
	}
};

EventHandler.prototype.handleRowsResetEvent = function(event) {
	var self = this;
	var promise = $.Deferred();
	if (NSDK_PROXY.pageAt(event.path) !== undefined) {
		NSDK_PROXY.pageAt(event.path).refresh()
			.done(function() {
				self.browsers.forEach(function(browser) {
					for (var i = 0; i < browser.page_stack.length; ++i) {
						if (event.path == browser.page_stack[i].getPath()) {
							var rowsCount = NSDK_PROXY.pageAt(event.path).rowsCount;
							if (rowsCount > 0) {
								browser.page_stack[i].getRows(0, NSDK_PROXY.pageAt(event.path).rowsCount)
									.done(function() {
										promise.resolve();
									});
								break;
							} else {
								promise.reject();
							}
						}
					}
				});
			});
	}
	return promise.promise();
};

EventHandler.prototype.handleRowsRemoveEvent = function(event) {
	this.browsers.forEach(function(browser) {
		for (var i = 0; i < browser.page_stack.length; ++i) {
			if (event.path == browser.page_stack[i].getPath()) {
				browser.page_stack[i].destroy();
				browser.page_stack.splice(i, 1);
				// decrement i because the stack size get decreased
				i--;
				if (i == browser.page_stack.length - 1) {
					// if the root page gets removed and it was one and only in the stack then there is no previous page to activate() (there is no browser.page_stack[-1] in the array)
					if (browser.page_stack.length > 0) {
						browser.page_stack[browser.page_stack.length - 1].activate();
					}
				}
			}
		}
	});
};

/**
 * If the "rowsType" event is "update" event the version of
 * the rows should be checked: if the difference between the
 * new and old versions equal the number of the subevents
 * (elements in event.rowsEvents array) then we have all events
 * delivered, thus we can start handling them. Otherwise we put
 * them into pending_updates array and wait for another "update"
 * event, then do the same procedure.
 */
EventHandler.prototype.handleRowsUpdateEvent = function(event) {
	var self = this;
	if (this.promise === undefined || this.promise.state() != "pending") {
		this.promise = $.Deferred();
	}
	var page;
	this.browsers.forEach(function(browser) {
		for (var i = 0; i < browser.page_stack.length; ++i) {
			if (browser.page_stack[i].getPath() == event.path) {
				page = browser.page_stack[i];
			}
		}
	});
	if (page === undefined) {
		return this.promise.promise();
	}
	var promises = [];
	var finished = false;
	this.pending_updates = this.pending_updates.concat(event.rowsEvents);

	for (var i = 0; i < this.pending_updates.length; ++i) {
		if (this.pending_updates[i].type == "remove") {
			if (i > 0 && this.pending_updates[i - 1].type != "remove" && this.pending_updates[i - 1].index > this.pending_updates[i].index) {
				this.pending_updates[i - 1].index -= 1;
			}
			page.removeRowAt(this.pending_updates[i].index);
		}
	}
	for (var i = 0; i < this.pending_updates.length; ++i) {
		if (this.pending_updates[i].type == "add") {
			promises.push(page.insertRowAt(this.pending_updates[i].index));
		} else if (this.pending_updates[i].type == "update") {
			promises.push(page.updateRowAt(this.pending_updates[i].index));
		}
	}
	self.pending_updates = [];
	$.when.apply($, promises)
		.done(function() {
			self.promise.resolve(event.path);
		})
		.fail(function() {
			self.promise.reject(event.path);
		});

	return this.promise.promise();
};

var NSDK_PROXY = (function($) {
	var paths = {};
	var references = {};
	var subscribes = {};
	var eventHandler = new EventHandler();

	function subscriptionType(path) {
		if (paths[path].get("type") == 'container') {
			return "rows";
		}
		return "item";
	}

	function _browse(path, ret) {
		var nsdkPath = null;
		if (paths[path]) {
			/* This NsdkPath already exists */
			references[path] += 1;
			ret.resolve(path);
			return ret.promise();
		}

		/* New NsdkPath is going to be opened */
		references[path] = 1;
		nsdkPath = new NsdkPath(path);
		paths[path] = nsdkPath;

		/* Load NsdkPath data before resolving the deferred */
		nsdkPath.refresh()
			.done(function(pathNew) {
				if (pathNew.error) {
					ret.reject(pathNew);
				} else {
					_browseRecursive(path, pathNew, ret);
				}
			});

		return ret.promise();
	}

	function _browseRecursive(pathOld, pathNew, ret) {
		if (pathNew && pathNew.rowsRedirect) {
			_browse(pathNew.rowsRedirect, ret)
				.done(function(result) {
					NSDK_PROXY.close(pathOld);
				});
		} else {
			subscribes[pathOld] = NSDK_Subscribe.subscribe(pathOld, subscriptionType(pathOld), function(event) {
				eventHandler.handle(event);
			});
			ret.resolve(pathOld);
		}
	}
	return {
		/**
		 * Create new NsdkPath instance, which respresents given path
		 * type:
		 * 		- container - performs getRows
		 * 		- item - only getData - for paths like (playTime, playData etc)
		 */
		browse: function(path) {
			return _browse(path, $.Deferred());
		},
		/**
		 * Closes NsdkContainer instance representing given path, invalidating all
		 * references to it.
		 */
		close: function(path) {
			if (!paths[path])
				return;
			references[path] -= 1;
			if (references[path] == 0) {
				NSDK_Subscribe.unsubscribe(subscribes[path]);
				delete subscribes[path];
				delete paths[path];
			}
		},
		pageAt: function(path) {
			return paths[path];
		},
		subscribe: function(path, eventType, handler) {
			if (paths[path]) {
				if (!eventHandler.subscriptions.hasOwnProperty(path)) {
					eventHandler.subscriptions[path] = {};
				}
				eventHandler.subscriptions[path][eventType] = handler;
				if (eventType == "update") {
					handler(paths[path]);
				}
			}
		},
		registerBrowser: function(browser) {
			if (eventHandler.browsers.indexOf(browser) == -1) {
				eventHandler.browsers.push(browser);
			}
		},
		deregisterBrowser: function(browser) {
			var index = eventHandler.browsers.indexOf(browser);
			if (index >= 0) {
				eventHandler.browsers.splice(index, 1);
			}
		}
	};
})(jQuery);

function NsdkPath(path) {
	var self = this;
	var path = path;
	this.extra_roles = [];
	self.path = path;
	self.initialized = false;

	var item_data = {};
	var listeners = [];

	var all_roles = ["title", "icon", "type", "containerType", "personType", "albumType",
		"imageType", "audioType", "videoType", "epgType", "modifiable",
		"disabled", "flags", "path", "value", "valueOperation()", "edit",
		"mediaData", "query", "activate", "likeIt", "rowsOperation", "setRoles",
		"timestamp", "id", "valueUnit", "context", "description",
		"longDescription", "search", "prePlay", "activity",
		"cancel", "accept", "risky", "preferred", "httpRequest", "encrypted",
		"encryptedValue", "rating", "fillParent", "autoCompletePath",
		"busyText", "sortKey", "renderAsButton", "doNotTrack", "persistentMetaData",
		"containerPlayable", "releaseDate"
	];

	/**
	 * Force refresh of all cached data
	 * type:
	 * 		- container - performs getRows
	 * 		- item - only getData - for paths like (playTime, playData etc)
	 */
	this.refresh = function() {
		var ret = jQuery.Deferred();
		var extra, count = 0;

		if (-1 != ["container", "query"].indexOf(self.get("type"))) {
			NSDK.getRows(path, all_roles.join(), 0, 0)
				.always(function(data) {
					updateRowsData(data);
					extra = self.get("accept");
					if (extra && extra != "'") {
						self.extra_roles.push("accept");
						++count;
					}
					ret.resolve(data)
				})
		} else { // type not yet known, or element's type requires getData only
			NSDK.getData(path, all_roles.join())
				.always(function(data) {
					if (!$.isArray(data) || data.error) {
						data = 0;
					}

					updateItemData(data);

					// this was an element with unknown type, check if getRows is required
					if (data == 0 || -1 != ["container", "query"].indexOf(self.get("type"))) {
						NSDK.getRows(path, all_roles.join(), 0, 0)
							.always(function(data) {
								updateRowsData(data);
								extra = self.get("accept");
								if (extra && extra != "'") {
									self.extra_roles.push("accept");
									++count;
								}
								ret.resolve(data)
							})
					} else {
						ret.resolve(data);
					}
				})
		}
		return ret.promise();
	}

	function updateItemData(data) {
		item_data = data
	}

	function updateRowsData(data) {
		if (data.error) {
			self.error = data.error;
			self.rowsCount = 0;
			self.rowsVersion = 0;
			self.rowsRedirect = 0;
			self.rowsRoles = 0;
		} else {
			self.error = 0;
			self.initialized = true;

			if (data.rowsCount != null)
				self.rowsCount = data.rowsCount;

			if (data.rowsVersion != null)
				self.rowsVersion = data.rowsVersion;

			if (data.rowsRedirect != null)
				self.rowsRedirect = data.rowsRedirect;

			/* These roles may actually differ from the ones returned from getData
			 * */
			if (data.roles != null)
				self.rowsRoles = data.roles
		}
	}

	function reformatRows(rows) {
		var ret = []
		for (var row in rows) {
			var ret_row = {}
			for (var i in all_roles) {
				if (rows[row][i] == null)
					continue
				ret_row[all_roles[i]] = rows[row][i]
			}
			ret.push(ret_row)
		}
		return ret;
	}

	/**
	 * Called from NsdkProxy when change event is reeived
	 */
	this.onChange = function(event) {
		/* A change has happened. If this is valid change event, first refresh
		 * all locally cached parameters, then invoke all change listeners. If
		 * event is not valid (e.g. communication error has occured), notify
		 * listeners.
		 */
		var type;
		console.log("Event: ", event);
		if (!$.isEmptyObject(event)) {
			if (!$.isEmptyObject(event.itemType))
				type = "item";
			else
				type = "container";

			return self.refresh(type).done(function() {
				for (var l in listeners) {
					listeners[l](self, "success");
				}
			})
		} else
			for (var l in listeners)
				listeners[l](self, "failed")
	}

	// Querying path properties

	/**
	 * Query a role of this path
	 */
	this.get = function(name) {
		var role_idx = $.inArray(name, all_roles)
		var ret = null

		if ((role_idx != -1) && (item_data[role_idx] != null)) {
			ret = item_data[role_idx];
		}

		/* Treat empty dictionary as null */
		if ((ret instanceof Object) && (Object.keys(ret).length === 0))
			ret = null

		if ((self.rowsRoles && self.rowsRoles[name]) && (ret == null))
			ret = self.rowsRoles[name]

		return ret;
	}

	/**
	 * Fetch container rows, starting at 'from' and ending at most at 'to' row.
	 */
	this.rows = function(from, to) {
		var ret = jQuery.Deferred();

		NSDK.getRows(path, all_roles.join(), from, to)
			.done(function(data) {
				updateRowsData(data)
				if (data.error)
					ret.reject(data.error)
				else
					ret.resolve(self.rowsCount, reformatRows(data.rows))
			})

		return ret.promise()
	}

	this.subscribe = function(f) {
		/* Invoke handler, so that external state is in sync */
		f(self, "success")
		listeners.push(f)
	}
}
