'use strict';

let controller = null;

/**
 * Represents a control.
 * @constructor
 */
class Control {
  constructor(view, $parent, config) {
    this.view = view;
    this.name = config.name;  // name of the control
    this.$parent = $parent;   // [DOM] parent container
    this.$container = null;   // [DOM] created container for the control
    this.$dom = null;
    this.$label = null;
    this.config = config;
    this.children = [];
    this.hasLabel = !(config['hasLabel'] === false);
    this.inlineControl = config['inlineControl'] === true;
  }

  /**
   *  Create a div and a label for this control.
   */
  createDOM() {
    this.$container =
        $('<div>').addClass('control-container').appendTo(this.$parent);
    if (this.hasLabel) {
      this.$label = $('<label>')
                        .addClass('control-label')
                        .attr({'for': this.config.name})
                        .html(this.config.name + ':')
                        .appendTo(this.$container);
    }
    return this;
  }

  remove() {
    this.$container.remove();
    return this;
  }

  createInputControl(type, id) {
    return $('<input>')
        .attr({'type': type, 'id': id, 'name': id})
        .addClass('control');
  }

  update(value) {}

  validate() {
    return false;
  }

  hasName(name) {
    return false;
  }

  onChanged() {
    this.view.onChanged();
  }

  exportJSON() {}
}

class ControlFactory {
  static createControl(type, view, $parent, config) {
    let controlClass = null;
    switch (type) {
      case 'text':
        controlClass = TextControl;
        break;
      case 'slide':
        controlClass = SlideControl;
        break;
      case 'composed':
        controlClass = ComposedControl;
        break;
      case 'checkbox':
        controlClass = CheckBoxControl;
        break;
      case 'select':
        controlClass = SelectControl;
        break;
      case 'template_list':
        controlClass = TemplateListControl;
        break;
      case 'template':
        controlClass = TemplateControl;
        break;
      case 'newline':
        controlClass = NewlineControl;
        break;
      case 'matrix':
        controlClass = MatrixControl;
        break;
      case 'biggie_orientation':
        controlClass = BiggieOrientationControl;
        break;
      case 'view_list':
        controlClass = ViewListControl;
        break;
      default:
        console.log('Unsupported type: ' + type);
        return null;
        break;
    }
    return new controlClass(view, $parent, config);
  }
}

class TextControl extends Control {
  constructor(view, $parent, config) {
    super(view, $parent, config);
    this.$text = null;
  }

  createDOM() {
    super.createDOM();
    let self = this;
    this.$text = super.createInputControl('text', this.config.name)
                     .val(this.config.default)
                     .appendTo(this.$container)
                     .change(function() {
                       self.onChanged();
                     });
    if (this.config.placeholder) {
      this.$text.attr({'placeholder': this.config.placeholder});
    }
  }

  validate() {
    return false;
  }

  val(value) {
    this.$text.val(value);
  }

  exportJSON() {
    let val = this.$text.val();

    if (this.config['is_array'] === true) {
      val = val.split(',');
      if (this.config['is_number'] === true) {
        return val.map(parseFloat);
      }
    }

    if (this.config['is_number'] === true) {
      return parseFloat(val);
    }

    return val;
  }
}

class SelectControl extends Control {
  constructor(view, $parent, config) {
    super(view, $parent, config);
  }

  createDOM() {
    super.createDOM();
    let self = this;
    this.$select = $('<select>').addClass('control').appendTo(this.$container);

    for (let i = 0; i < this.config.options.length; ++i) {
      let optionConfig = this.config.options[i];
      $('<option>', {value: optionConfig['value'], text: optionConfig['text']})
          .appendTo(this.$select);
    }

    this.$select.val(this.config.default);
    this.$select.change(function() {
      self.onChanged();
    });
  }

  validate() {
    return false;
  }

  val(value) {
    this.$select.val(value);
  }

  exportJSON() {
    let val = this.$select.val();
    if (this.config['is_number'] === true) {
      return parseFloat(val);
    }
    return val;
  }
}

class CheckBoxControl extends Control {
  constructor(view, $parent, config) {
    super(view, $parent, config);
    this.$text = null;
  }

  createDOM() {
    super.createDOM();
    let self = this;
    this.$dom = super.createInputControl('checkbox', this.config.name)
                    .prop('checked', this.config.default)
                    .appendTo(this.$container)
                    .change(function() {
                      self.onChanged();
                    });
  }

  val(value) {
    this.$dom.prop('checked', value);
  }

  validate() {
    return false;
  }

  exportJSON() {
    return this.$dom.prop('checked');
  }
}

class SlideControl extends Control {
  constructor(view, $parent, config) {
    super(view, $parent, config);
    this.$dom = null;
  }

  createDOM() {
    super.createDOM();
    let self = this;

    let slideRange = this.config.range;
    let slideDefaultValue = this.config.default;
    if (this.config.is_log == true) {
      slideRange = {
        'min': Math.log10(this.config.range.min),
        'max': Math.log10(this.config.range.max),
        'step': Math.max(1e-5, Math.log10(this.config.range.step))
      };
      slideDefaultValue = Math.log10(slideDefaultValue);
    }

    this.$dom = super.createInputControl('range', this.config.name)
                    .attr(slideRange)
                    .val(slideDefaultValue)
                    .appendTo(this.$container)
                    .addClass('slide')
                    .on('input', function() {
                      let raw_value = $(this).val();
                      let value = raw_value;
                      if (self.config.is_log === true) {
                        let frac = Math.min(0, Math.log10(self.config.range.step));
                        value = Math.pow(10, raw_value).toFixed(-frac);
                      }
                      let inputValue = parseFloat(self.$value.val());
                      if (inputValue != value) {
                        self.$value.val(value);
                        self.onChanged();
                      }
                    });
    this.$value = $('<input>')
                      .attr({'id': this.config.name + '-value'})
                      .addClass('slide-value')
                      .val(this.config.default)
                      .appendTo(this.$container)
                      .on('change', function() {
                        let value = parseFloat($(this).val());
                        value = Math.min(value, self.config.range.max);
                        value = Math.max(value, self.config.range.min);
                        self.setSlideValue(value);
                        $(this).val(value);
                        self.onChanged();
                      });
    this.$unit = $('<span>').html(this.config.unit).appendTo(this.$container);
  }

  val(value) {
    this.setSlideValue(value);
    this.$value.val(value);
  }

  setSlideValue(inputValue) {
    if (this.config.is_log === true) {
      this.$dom.val(Math.log10(inputValue));
    } else {
      this.$dom.val(inputValue);
    }
  }

  validate() {
    return false;
  }

  exportJSON() {
    let val = this.$value.val();
    if (this.config['is_number'] === true) {
      return parseFloat(val);
    }
    return val;
  }
}

class ComposedControl extends Control {
  constructor(view, $parent, config) {
    super(view, $parent, config);
    this.$dom = null;
  }

  createDOM() {
    super.createDOM();
    this.$dom =
        $('<div>').addClass('composed-control-body').appendTo(this.$container);
    for (let i = 0; i < this.config.controls.length; ++i) {
      let controlConfig = this.config.controls[i];
      let control = ControlFactory.createControl(
          controlConfig.type, this.view, this.$dom, controlConfig);
      if (control === null) continue;
      this.children.push(control);
      control.createDOM();
      if (this.inlineSubcontrols) {
        control.$container.addClass('control-container-inline');
        if (control.$label != null) {
          control.$label.removeClass('control-label')
              .addClass('control-label-inline');
        }
      }
    }
  }

  val(value) {
    for (let key in value) {
      for (let i = 0; i < this.children.length; ++i) {
        if (this.children[i].config.name == key)
          this.children[i].val(value[key]);
      }
    }
  }

  validate() {
    return false;
  }

  exportJSON() {
    let json = {};
    for (let i = 0; i < this.children.length; ++i) {
      let child = this.children[i];
      json[child.config.name] = child.exportJSON();
    }
    return json;
  }
}

class NewlineControl extends Control {
  constructor(view, $parent, config) {
    super(view, $parent, config);
    this.$dom = null;
    this.hasLabel = false;
    this.ininlineControl = false;
  }

  createDOM() {
    super.createDOM();
    this.$container.html('<br>');
  }
}

class MatrixControl extends ComposedControl {
  constructor(view, $parent, config) {
    super(view, $parent, config);
    this.inlineSubcontrols = true;
    this.expendConfig();
  }

  expendConfig() {
    this.config.controls = [];
    for (let i = 0; i < this.config.rows; ++i) {
      for (let j = 0; j < this.config.cols; ++j) {
        let controlConfig = {
          'name': 'cell_' + i + '_' + j,
          'row': i,
          'col': j,
          'type': 'text',
          'is_number': true,
          'default': this.config.default[i][j],
          'hasLabel': false
        };
        this.config.controls.push(controlConfig);
      }
      let newlineConfig = {'type': 'newline'};
      this.config.controls.push(newlineConfig);
    }
  }

  setValue(row, col, value) {
    for (let i = 0; i < this.children.length; ++i) {
      let config = this.children[i].config;
      if (config.row == row && config.col == col) {
        this.children[i].val(value);
        return;
      }
    }
  }

  val(value) {
    for (let i = 0; i < value.length; ++i)
      for (let j = 0; j < value[0].length; ++j)
        this.setValue(i, j, value[i][j]);
  }

  validate() {
    return false;
  }

  exportJSON() {
    let arr = [];
    for (let i = 0; i < this.config.rows; ++i) {
      arr.push([]);
      for (let j = 0; j < this.config.cols; ++j) {
        arr[i].push(0);
      }
    }

    for (let i = 0; i < this.children.length; ++i) {
      let config = this.children[i].config;
      if (config.row >= 0 && config.col >= 0) {
        arr[config.row][config.col] = this.children[i].exportJSON();
      }
    }

    return arr;
  }
}

class TemplateControl extends ComposedControl {
  constructor(view, $parent, config) {
    super(view, $parent, config);
    this.hasLabel = false;
    this.inlineSubcontrols = config['inline'] !== false;
  }

  validate() {
    return false;
  }
}

class TemplateListControl extends Control {
  constructor(view, $parent, config) {
    super(view, $parent, config);
    this.$dom = null;
  }

  createDOM() {
    let self = this;
    super.createDOM();
    this.$dom =
        $('<div>').addClass('template-list-body').appendTo(this.$container);

    if (this.config.default) {
      for (let i = 0; i < this.config.default.length; ++i) {
        this.addControl(this.config.default[i]);
      }
    }

    this.$addButtonContainer = $('<div>').appendTo(this.$container);
    this.$addButton = $('<button>')
                          .html('Add')
                          .appendTo(this.$addButtonContainer)
                          .click(function() {
                            self.addControl();
                          });
  }

  // Create control from template.
  addControl(value) {
    let self = this;
    let control = ControlFactory.createControl(
        'template', this.view, this.$dom, this.config['template']);
    if (control === null) return;
    this.children.push(control);
    control.createDOM();
    control.$removeButton = $('<button>')
                                .addClass('template-remove-button')
                                .html('Remove')
                                .appendTo(control.$dom)
                                .click(function() {
                                  self.removeControl(control);
                                });
    if (this.config.tunable !== true) {
      controller.tunable = false;
    }
    if (value) control.val(value);
  }

  removeControl(control) {
    let index = this.children.indexOf(control);
    this.children.splice(index, 1);
    control.remove();
    if (this.config.tunable === true) {
      this.view.onChanged();
    } else {
      controller.tunable = false;
    }
  }

  val(value) {
    // Remvoe all the controls.
    while (this.children.length > 0) {
      this.removeControl(this.children[0]);
    }

    for (let i = 0; i < value.length; ++i) {
      this.addControl(value[i]);
    }
  }

  validate() {
    return false;
  }

  exportJSON() {
    let json = [];
    for (let i = 0; i < this.children.length; ++i) {
      let child = this.children[i];
      json.push(child.exportJSON());
    }
    return json;
  }
}

class BiggieOrientationControl extends ComposedControl {
  constructor(view, $parent, config) {
    super(view, $parent, config);
    this.$dom = null;
    this.expendConfig();
  }

  expendConfig() {
    const kRows = 2;
    const kCols = 2;
    this.config.controls = [];
    const names = ['low', 'high'];
    for (let i = 0; i < names.length; ++i) {
      let matrixConfig = {
        'name': names[i],
        'type': 'matrix',
        'rows': kRows,
        'cols': kCols,
        'default': this.config.default[names[i]]
      };
      this.config.controls.push(matrixConfig);
    }
  }
}

class ViewListControl extends Control {
  constructor(view, $parent, config) {
    super(view, $parent, config);
    this.$dom = null;
    this.parentProcessor = view;
  }

  createDOM() {
    let self = this;
    super.createDOM();
    this.$dom = $('<div>').appendTo(this.$container);
    this.viewList = new ViewList(
        this.$dom, this.config['list'], Object.keys(this.config['list']));
    this.viewList.isSubprocessor = true;
    this.viewList.parentProcessor = this.parentProcessor;
    this.viewList.createDOM();
  }

  hasName(name) {
    return this.viewList.hasName(name);
  }

  val(value) {
    this.viewList.removeAllViews();
    for (let i = 0; i < value.length; ++i) {
      this.viewList.addView(value[i]['name'], value[i]['type'])
          .val(value[i]['config']);
    }
  }

  exportJSON() {
    let json = this.viewList.exportJSON();
    return json['processors'].map(x => {
      x['type'] = x['processor'];
      delete x['processor'];
      return x;
    });
  }
}

class ProcessorList {
  constructor($parent, config, avail_processors, onAdd) {
    this.$parent = $parent;
    this.config = config;
    this.avail_processors = avail_processors;
    this.onAdd = onAdd;
  }

  createDOM() {
    let self = this;
    let $body = $('<div>').appendTo(this.$parent);
    $('<span>').html('type:').appendTo($body);
    let $select = $('<select>').appendTo($body);

    let processors = [];
    for (let type in this.config) {
      // e.g. "_comment"
      if (type[0] == '_') continue;
      if (this.avail_processors.indexOf(type) !== -1) {
        processors.push(type);
      }
    }

    processors.sort().forEach(function(type) {
      $select.append($('<option>', {value: type, text: type}));
    });

    $('<span>').html('name:').appendTo($body);
    let $text = $('<input>').attr({'type': 'text'}).appendTo($body);

    $('<button>').html('Add').appendTo($body).click(function() {
      let name = $text.val();
      if (name.length == 0) {
        alert('name can\'t be empty!');
        return;
      }
      if (controller.hasName(name)) {
        controller.showError(
            'name: "' + name + '" already used! Try a different name.');
        return;
      }
      let type = $select.val();
      $text.val('');
      if (self.onAdd) self.onAdd(name, type);
    });

    $select.change(function(){
      let type = $select.val();
      $text.val(self.config[type].name || '');
      $text.prop("disabled", self.config[type].fixed_name === true);
    });
  }
}

class View {
  /**
   * Create a view.
   * @param {string} name - Name of the view.
   * @param {string} processor - Type / module of the view.
   * @param {object} $parent - dom object to append to.
   * @param {object} config - config object to create the view from.
   */
  constructor(name, processor, $parent, config) {
    this.name = name;
    this.processor = processor;
    this.$parent = $parent;
    this.config = config;
    this.controls = [];  // list of controls.
    this.timeoutId = null;
    this.visible_ = true;
  }

  createDOM() {
    let self = this;
    this.$container = $('<div>').addClass('view').appendTo(this.$parent);
    this.$header = $('<div>').addClass('view-header').appendTo(this.$container);
    this.$name = $('<input type="text">')
                     .val(this.name)
                     .addClass('view-name')
                     .appendTo(this.$header)
                     .change(function() {
                       self.rename($(this).val());
                     });

    this.$processorType = $('<span>')
                              .addClass('view-title')
                              .html('[' + this.processor + ']')
                              .appendTo(this.$header);

    this.$removeButton = $('<button>')
                             .addClass('view-action-button')
                             .html('Remove')
                             .appendTo(this.$header)
                             .click(function() {
                               self.onRemoved();
                             });

    this.$toggleButton = $('<button>')
                             .addClass('view-action-button')
                             .addClass('hide-button')
                             .html('Hide')
                             .appendTo(this.$header)
                             .click(function() {
                               self.visible = !self.visible;
                             });

    this.$body = $('<div>').addClass('view-body').appendTo(this.$container);
    for (let i = 0; i < this.config.controls.length; ++i) {
      let control = ControlFactory.createControl(
          this.config.controls[i].type, this, this.$body,
          this.config.controls[i]);
      if (control === null) continue;
      control.createDOM();
      this.controls.push(control);
    }
  }

  rename(name) {
    if (name != this.name && controller.hasName(name)) {
      controller.showError('Name "' + name + '" already used!');
      this.$name.val(this.name);
      return;
    }
    this.name = name;
    this.$name.html(this.name);
    controller.tunable = false;
  }

  // Hide the view.
  hide() {
    this.visible = false;
  }

  // Show the view.
  show() {
    this.visible = true;
  }

  get visible() {
    return this.visible_;
  }

  set visible(val) {
    if (this.visible_ == val) {
      return;
    }
    this.visible_ = val;
    if (val) {
      this.$body.slideToggle('fast');
      this.$toggleButton.html('Hide');
    } else {
      this.$body.slideToggle('fast');
      this.$toggleButton.html('Show');
    }
  }

  // Update the value for its controls.
  val(value) {
    if (value['_comment']) {
      this.comment = value['_comment'];
    }
    for (let key in value) {
      for (let i = 0; i < this.controls.length; ++i) {
        if (this.controls[i].name == key) {
          if (typeof this.controls[i].val === 'function') {
            this.controls[i].val(value[key]);
          } else {
            console.log('control.val is not implemented');
            console.log(this.controls[i].exportJSON());
          }
          break;
        }
      }
    }

    // Restore the view state if given.
    if (value['_view_opened'] != null) {
      this.visible = value['_view_opened'];
    }
  }

  // Returns the UI index (order) in a ViewList.
  index() {
    return this.$container.index();
  }

  remove() {
    this.$container.remove();
  }

  hasName(name) {
    for (let i = 0; i < this.controls.length; ++i) {
      if (this.controls[i].hasName(name)) return true;
    }
    return this.name === name;
  }

  _sendUpdateMessage() {
    let json = this.exportJSON();
    let data = {};
    data['name'] = this.isSubprocessor ? this.parentProcessor.name : this.name;
    let config = json['config'];
    data['message'] = this.isSubprocessor ?
        {'subprocessor': this.name, 'config': config} :
        config;
    const url = '/setup/audio/set_params';
    $.ajax({
      type: 'POST',
      url: url,
      contentType: 'application/json',
      data: JSON.stringify(data),
      success: function(r) {
        console.log('set_params success!' + r);
      },
      dataType: 'json'
    });
  }

  onChanged() {
    if (this.config.tunable === false) {
      controller.tunable = false;
      return;
    }

    // Do not send update messages when a write config must be performed.
    if (!controller.tunable) return;

    const sendMessageDelay = 50;  // in milliseconds.

    if (this.timeoutId !== null) {
      clearTimeout(this.timeoutId);
    }

    let self = this;

    this.timeoutId = setTimeout(function() {
      self._sendUpdateMessage();
    }, sendMessageDelay);
  }

  exportJSON(includeViewState) {
    let json = {};
    let config = {};
    let processorKey = 'processor';
    let nameKey = 'name';
    if (this.processor.indexOf('2.0') != -1) {
      processorKey = 'lib';
    }
    json[processorKey] = this.processor;

    if (this.name) {
      json[nameKey] = this.name;
    }

    if (this.comment) {
      json['_comment'] = this.comment;
    }

    for (let i = 0; i < this.controls.length; ++i) {
      let control = this.controls[i];
      let control_json = control.exportJSON();
      // Skip if config is optional and value is invalid.
      if (control.config.is_optional && !control_json) {
        continue;
      }
      config[control.config.name] = control_json;
    }

    // Store the state of this view into config to later restoration.
    if (includeViewState === true) {
      config['_view_opened'] = this.visible_;
    }

    json['config'] = config;

    return json;
  }
}


class VolumeMap extends View {
  constructor($parent, config) {
    super('volume_map', 'volume_map', $parent, config);
  }

  createDOM() {
    super.createDOM();
    // Unremoveable.
    this.$removeButton.hide();
  }

  onChanged() {
    controller.tunable = false;
    return;
  }

  exportJSON(includeViewState) {
    return super.exportJSON(includeViewState)['config']['volume_map'];
  }
}

class DefaultVolume extends View {
  constructor($parent, config) {
    super('default_volume', 'default_volume', $parent, config);
  }

  createDOM() {
    super.createDOM();
    // Unremoveable.
    this.$removeButton.hide();
  }

  onChanged() {
    controller.tunable = false;
    return;
  }

  exportJSON(includeViewState) {
    let json = {'dbfs': super.exportJSON(includeViewState)['config']};
    return json;
  }
}

// A list of views.
class ViewList {
  constructor($parent, config, avail_processors) {
    this.$parent = $parent;
    this.views = [];
    this.config = config;
    this.avail_processors = avail_processors;
    this.isSubprocessor = false;
    this.parentProcessor = null;
  }

  createDOM() {
    let self = this;

    this.$body = $('<div>').appendTo(this.$parent);

    if (!this.isSubprocessor) {
      let $streamsSection = $('<p>')
                                .html('Input Streams: ')
                                .appendTo(this.$body)
                                .on('change', function() {
                                  controller.tunable = false;
                                });

      this.$streams =
          $('<input>')
              .attr({'type': 'text', 'id': 'streams', 'name': 'streams'})
              .addClass('control')
              .appendTo($streamsSection);

      this.$numInputChannelsSection = $('<p>')
                                          .html('Number of input channels: ')
                                          .appendTo(this.$body)
                                          .hide();
      // A read-only field.
      this.$num_input_channels = $('<input>')
                                     .attr({
                                       'type': 'text',
                                       'id': 'num_input_channels',
                                       'name': 'num_input_channels'
                                     })
                                     .val(-1)
                                     .prop('disabled', true)
                                     .addClass('control')
                                     .appendTo(this.$numInputChannelsSection);

      this.$volumeLimitsSection = $('<p>')
                                      .html('Volume Limits: ')
                                      .appendTo(this.$body);
      this.$volumeLimitsDefaultSection = $('<div>')
                                      .html('Default: ')
                                      .appendTo(this.$volumeLimitsSection);
      this.$volumeLimitsDefaultMinSection = $('<span>')
                                      .html('Min: ')
                                      .appendTo(this.$volumeLimitsDefaultSection);
      this.$volumeLimitsDefaultMaxSection = $('<span>')
                                      .html('Max: ')
                                      .appendTo(this.$volumeLimitsDefaultSection);
      this.$volume_limits_min =
          $('<input>')
              .attr({'type': 'text', 'id': 'volume_limits_min', 'name': 'volume_limits_min'})
              .addClass('control')
              .appendTo(this.$volumeLimitsDefaultMinSection);
      this.$volume_limits_max =
          $('<input>')
              .attr({'type': 'text', 'id': 'volume_limits_max', 'name': 'volume_limits_max'})
              .addClass('control')
              .appendTo(this.$volumeLimitsDefaultMaxSection);
    }

    let $removeSection = $('<p>').appendTo(this.$body);

    let $removeAllButton = $('<button>')
                               .html('Remove All Processors')
                               .appendTo($removeSection)
                               .click(function() {
                                 while (self.views.length > 0) {
                                   self.views[0].remove();
                                   self.views.splice(0, 1);
                                   controller.tunable = false;
                                 }
                               });

    this.$processorContainer = $('<div>').appendTo(this.$body);

    this.$processorContainer.sortable({
      change: function(event, ui) {
        controller.tunable = false;
      }
    });

    this.$processorContainer.disableSelection();

    let $newProcessorSection = $('<p>').appendTo(this.$body);

    this.processorList = new ProcessorList(
        $newProcessorSection, this.config, this.avail_processors,
        function(name, type) {
          self.addView(name, type);
        });

    this.processorList.createDOM();
  }

  addView(name, type) {
    let self = this;
    let $parent = this.$processorContainer;
    let view = new View(name, type, $parent, this.config[type]);
    view.onRemoved = function(event) {
      let index = self.views.indexOf(view);
      self.views.splice(index, 1);
      view.remove();
      controller.tunable = false;
    };

    view.isSubprocessor = this.isSubprocessor;
    view.parentProcessor = this.parentProcessor;

    view.createDOM();
    this.views.push(view);
    controller.tunable = false;
    return view;
  }

  removeAllViews() {
    while (this.views.length > 0) {
      this.views[0].remove();
      this.views.splice(0, 1);
      controller.tunable = false;
    }
  }

  setStreams(streams) {
    if (streams) {
      this.$streams.val(streams.join(','))
    }
  }

  setNumInputChannels(num_input_channels) {
    if (num_input_channels && num_input_channels >= 1) {
      this.$num_input_channels.val(num_input_channels);
      this.$numInputChannelsSection.show();
    }
  }

  setVolumeLimits(volume_limits) {
    this.volume_limits = volume_limits || {};
    let limit_min = this.volume_limits.min || 0.0;
    let limit_max = this.volume_limits.max || 1.0;
    this.$volume_limits_min.val(limit_min);
    this.$volume_limits_max.val(limit_max);
  }

  remove() {}

  // Check whether any view has the give name;
  hasName(name) {
    for (var i = 0; i < this.views.length; ++i) {
      if (this.views[i].hasName(name)) {
        return true;
      }
    }
    return false;
  }

  updateVolumeLimits() {
    if (!this.volume_limits) {
      return;
    }

    if (this.$volume_limits_min) {
      this.volume_limits.min = parseFloat(this.$volume_limits_min.val());
    }
    if (this.$volume_limits_max) {
      this.volume_limits.max = parseFloat(this.$volume_limits_max.val());
    }

    if (this.volume_limits.hasOwnProperty("min") && this.volume_limits.min <= 0.0) {
      delete this.volume_limits.min;
    }
    if (this.volume_limits.hasOwnProperty("max") && this.volume_limits.max >= 1.0) {
      delete this.volume_limits.max;
    }
  }

  exportJSON(includeViewState) {
    // Sort views by their UI order.
    this.views.sort(function(viewA, viewB) {
      return viewA.index() - viewB.index();
    });

    let json = {};
    if (this.$streams) {
      json['streams'] = this.$streams.val().split(',').filter(function(el) {
        return el.length != 0;
      });
    }

    this.updateVolumeLimits();
    if (this.volume_limits && Object.entries(this.volume_limits).length > 0) {
      json["volume_limits"] = this.volume_limits;
    }

    if (this.$num_input_channels) {
      let num_input_channels = parseFloat(this.$num_input_channels.val());
      // Only set this field if it is present in the cast_audio.json.
      if (num_input_channels > 0) {
        json['num_input_channels'] = num_input_channels;
      }
    }
    json['processors'] = [];
    for (let i = 0; i < this.views.length; ++i) {
      json['processors'].push(this.views[i].exportJSON(includeViewState));
    }
    return json;
  }
}


class Linearize extends ViewList {
  constructor($parent, config, avail_processors) {
    super($parent, config, avail_processors);
  }

  // Overwrite the format.
  exportJSON(includeViewState) {
    let json = super.exportJSON(includeViewState);
    for (let i = 0; i < json.processors.length; ++i) {
      let processor = json.processors[i];
    }
    return json;
  }
}

class OutputStream {
  constructor(name, config, avail_processors) {
    this.name = name;
    this.$container = $('#stream-processors-container');
    this.$tabs = $('#stream-processors-tabs');
    this.config = config;
    this.avail_processors = avail_processors;
  }

  createDOM() {
    let self = this;
    let tabId = 'tab-' + this.name;
    this.$tab = $('<li>').appendTo(this.$tabs);
    let $href = $('<a>')
                    .attr('href', '#' + tabId)
                    .html(this.name)
                    .appendTo(this.$tab);
    this.$body = $('<div>').attr('id', tabId).appendTo(this.$container);

    this.viewList =
        new ViewList(this.$body, this.config, this.avail_processors);
    this.viewList.createDOM();

    let $removeSection = $('<p>').appendTo(this.$body);

    let name = self.name
    this.$removeButton = $('<button>')
                             .html('Remove Stream')
                             .appendTo($removeSection)
        .click(function() {
                               controller.removeOutputStream(name);
                             });

    // Open the current tab.
    this.$container.tabs('refresh');
    $href.click();
  }

  // Add a new view.
  // Returns the added view.
  addView(name, type) {
    return this.viewList.addView(name, type);
  }

  remove() {
    this.$body.remove();
    this.$tab.remove();
    this.$container.tabs('refresh');
    controller.tunable = false;
  }

  hasName(name) {
    return this.viewList && this.viewList.hasName(name);
  }

  exportJSON(includeViewState) {
    let json = this.viewList.exportJSON(includeViewState);
    json['name'] = this.name;
    return json;
  }
}


class AudioTuningController {
  constructor(config, avail_processors) {
    this.config = config;
    this.avail_processors = avail_processors;
    this.outputStreams = [];  // Array of OutputStream
    this.mix = null;          // ViewList
    this.linearize = null;    // ViewList
    this.configUrl = '/setup/audio/get_config';
    this.demoConfigUrl = '/data/cast_audio.json';
    this.defaultConfigUrl = '/setup/audio/get_default_config';
    this.tunable_ =
        false;  // real-time tunable, if false write config required.
  }

  get tunable() {
    return this.tunable_;
  }

  set tunable(val) {
    let cssClass = 'write-required';
    this.tunable_ = val;
    let $header = $('#header');
    $('#write-hint').show();
    if (val === true) {
      $('#write-hint').hide();
      $header.removeClass(cssClass);
    } else {
      $header.addClass(cssClass);
    }
  }

  // Add a new output stream.
  addOutputStream(name) {
    let outputStream =
        new OutputStream(name, this.config, this.avail_processors);
    outputStream.createDOM();
    this.outputStreams.push(outputStream);
    controller.tunable = false;
    return outputStream;
  }

  // Remove a output stream by stream IDs.
  removeOutputStream(name) {
    for (let i = 0; i < this.outputStreams.length; ++i) {
      let outputStream = this.outputStreams[i];
      if (outputStream.name == name) {
        outputStream.remove();
        this.outputStreams.splice(i, 1);
        controller.tunable = false;
        return;
      }
    }
    console.log("Error: could not remove output stream '" + name + "'")
  }

  removeAllOutputStreams() {
    for (let i =0; i < this.outputStreams.length; ++i) {
      this.outputStreams[i].remove();
    }
    this.outputStreams = []
    controller.tunable = false;
  }

  init() {
    this.mix = new ViewList(
        $('#mix-processors-body'), this.config, this.avail_processors);
    this.mix.createDOM();

    this.linearize = new Linearize(
        $('#linearize-processors-body'), this.config, this.avail_processors);
    this.linearize.createDOM();

    this.defaultVolume =
        new DefaultVolume($('#global-body'), this.config['default_volume']);
    this.defaultVolume.createDOM();

    this.createVolumeMapView();

    $('#stream-processors-container').tabs();

    $('#btn-add-stream').click(function() {
      let name = $('#txt-stream-names').val();
      if (name.length == 0) {
        alert('Stream names can\'t be empty!');
        return;
      }
      let names = name.split(',').map(s => s.trim());
      let stream = controller.addOutputStream(names);
      stream.viewList.setStreams(names);

      let channels = $('#txt-stream-channels').val().trim();
      if (channels.length > 0) {
        stream.viewList.setNumInputChannels(parseInt(channels, 10));
      }
      $('#txt-stream-names').val('');
      $('#txt-stream-channels').val('');
    });

    $('#btn-remove-all-streams').click(function() {
      while (controller.outputStreams.length > 0) {
        controller.removeOutputStream(controller.outputStreams[0].name);
      }
    });

    $('#preview-dialog')
        .dialog({autoOpen: false, modal: true, height: 750, width: 1000});
  }

  // Check whether any view has the given name.
  hasName(name) {
    if (this.mix.hasName(name) || this.linearize.hasName(name) ||
        name === this.volumeMap.name || name === this.defaultVolume.name) {
      return true;
    }

    for (let i = 0; i < this.outputStreams.length; ++i) {
      if (this.outputStreams[i].hasName(name)) {
        return true;
      }
    }

    return false;
  }

  // Pre-load a set of views for demo purpose.
  createDemoViews() {
    let stream1 = this.addOutputStream(['default', 'test']);
    let stream2 = this.addOutputStream(['communications']);

    stream1.addView('equalizer', 'libcast_equalizer_1.0.so');
    stream1.addView('room_equalizer', 'libcast_equalizer_1.0.so');
    stream1.addView('boomcloud', 'libcast_boomcloud_1.0.so');
    stream1.addView('waves', 'libcast_waves_1.0.so');

    stream2.addView('saturated_gain', 'libcast_saturated_gain_2.0.so');

    this.mix.addView('delay', 'libcast_delay_2.0.so');
    this.linearize.addView('equalizer', 'libcast_equalizer_2.0.so');
    this.linearize.addView('crossover', 'libcast_crossover_2.0.so');
    this.linearize.addView('orientation', 'libcast_orientation_biggie_1.0.so');
  }

  createVolumeMapView(values) {
    if (this.volumeMap) {
      this.volumeMap.remove();
    }

    let config = this.config['volume_map'];
    if (values) config['controls'][0]['default'] = values.slice();

    this.volumeMap = new VolumeMap($('#global-body'), config);
    this.volumeMap.createDOM();
  }

  createViewsFromConfig(config) {
    this.postProcessorsComment = config.postprocessors['_comment'];
    // Create output streams.
    let outputStreams = config.postprocessors.output_streams;
    for (let i = 0; i < outputStreams.length; ++i) {
      let outputStream = outputStreams[i];
      let name = outputStream.streams.join('-')
      if (outputStream['name']) {
        name = outputStream['name'];
      }
      let stream = this.addOutputStream(name);
      stream.viewList.setStreams(outputStream.streams);
      stream.viewList.setNumInputChannels(outputStream.num_input_channels);
      stream.viewList.setVolumeLimits(outputStream.volume_limits);
      for (let j = 0; j < outputStream.processors.length; ++j) {
        let processor = outputStream.processors[j];
        let view = stream.addView(
            processor.name, processor.processor || processor.lib);
        // Sync the value of the view with config.
        if (processor['_comment']) view.comment = processor['_comment'];
        view.val(processor.config);
      }
    }

    // Open the first tab.
    $('.ui-tabs-anchor').get(0).click();

    // Create linearize processors.
    this.linearize.setStreams(config.postprocessors.linearize.streams);
    this.linearize.setNumInputChannels(
        config.postprocessors.linearize.num_input_channels);
    this.linearize.setVolumeLimits(config.postprocessors.linearize.volume_limits);
    let linearizeProcessors = config.postprocessors.linearize.processors;
    for (let i = 0; i < linearizeProcessors.length; ++i) {
      let processor = linearizeProcessors[i];
      let view = this.linearize.addView(
          processor.name, processor.processor || processor.lib);
      if (processor['_comment']) view.comment = processor['_comment'];
      view.val(processor.config);
    }

    // Create mix processors.
    this.mix.setStreams(config.postprocessors.mix.streams);
    this.mix.setNumInputChannels(config.postprocessors.mix.num_input_channels);
    this.mix.setVolumeLimits(config.postprocessors.mix.volume_limits);
    let mixProcessors = config.postprocessors.mix.processors;
    for (let i = 0; i < mixProcessors.length; ++i) {
      let processor = mixProcessors[i];
      let view = this.mix.addView(
          processor.name, processor.processor || processor.lib);
      if (processor['_comment']) view.comment = processor['_comment'];
      view.val(processor.config);
    }

    // Remove and create a new volume map is easier...
    this.createVolumeMapView(config['volume_map']);

    this.defaultVolume.val(config['default_volume']['dbfs']);
  }

  clear() {
    this.removeAllOutputStreams();
    this.linearize.removeAllViews();
    this.mix.removeAllViews();
  }

  storeConfig(key) {
    localStorage.setItem(
        'config_' + key, JSON.stringify(this.exportJSON(true)));
    localStorage.setItem(
        'active_tab_index',
        $('#stream-processors-container').tabs('option', 'active'));
    this.showInfo(key + ' stored');
  }

  restoreConfig(key) {
    let configStr = localStorage.getItem('config_' + key);
    if (!configStr) {
      this.showInfo('Config not found!');
      return;
    }

    let config = JSON.parse(configStr);

    let self = this;
    this.applyConfig(config);
    // Open the active tab.
    $('#stream-processors-container')
        .tabs(
            'option', 'active',
            parseFloat(localStorage.getItem('active_tab_index')));
    this.volumeMap.hide();
    this.writeConfig(function() {
      self.showInfo(key + ' applied!');
    });
  }

  hideAll() {
    $('.hide-button').click();
  }

  applyConfig(config) {
    this.clear();
    this.createViewsFromConfig(config);
    this.tunable = true;
  }

  _loadConfig(url) {
    let self = this;
    $.getJSON(url, function(config) {
      self.applyConfig(config);
      self.showInfo('Config loaded!');
      self.hideAll();
    });
  }

  loadConfig() {
    this._loadConfig(this.configUrl);
  }

  showInfo(message) {
    $('#notification').html(message).stop(true, true).show().fadeOut(5000);
  }

  showError(message) {
    this.showInfo('<span style="color:red">' + message + '</span>');
  }

  loadDefaultConfig() {
    this._loadConfig(this.defaultConfigUrl);
  }

  loadDemoConfig() {
    this._loadConfig(this.demoConfigUrl);
  }

  writeConfig(successCallback, errorCallback) {
    let self = this;
    let config_obj = this.exportJSON()
    if (!this.validateJSON(config_obj)) {
      return
    }
    let config = stringify(config_obj, {maxLength: 80});
    const url = '/setup/audio/set_config';

    $.ajax({
      url: url,
      type: 'post',
      data: config,
      contentType: 'application/json',
      success: function(data) {
        self.tunable = true;
        self.showInfo('Config wrote!');
        if (successCallback) successCallback();
      },
      complete: function(jqXHR, status) {
        if (jqXHR.status != 200) {
          self.showInfo(jqXHR.responseText);
          if (errorCallback) errorCallback();
        }
      }
    });
  }

  deleteConfig(successCallback, errorCallback) {
    let self = this;
    const url = '/setup/audio/delete_config';

    $.ajax({
      url: url,
      type: 'post',
      data: '',
      contentType: 'text/plain',
      success: function(data) {
        self.tunable = true;
        self.showInfo('Config deleted!');
        if (successCallback) successCallback();
      },
      complete: function(jqXHR, status) {
        if (jqXHR.status != 200) {
          self.showInfo(jqXHR.responseText);
          if (errorCallback) errorCallback();
        }
      }
    });
  }

  exportJSON(includeViewState) {
    let json = {};
    if (this.postProcessorsComment) {
      json['_comment'] = this.postProcessorsComment;
    }

    json['output_streams'] =
        this.outputStreams.map(os => os.exportJSON(includeViewState));

    json['mix'] = this.mix.exportJSON(includeViewState);
    json['linearize'] = this.linearize.exportJSON(includeViewState);
    let output = {
      'postprocessors': json,
      'volume_map': this.volumeMap.exportJSON(includeViewState),
      'default_volume': this.defaultVolume.exportJSON(includeViewState)
    };
    return output;
  }

  validateJSON(json) {
    let valid_stream_types = ["default", "communications", "local", "platform",
                              "assistant-alarm", "assistant-tts", "bypass", "no_delay"]
    for (let i = 0; i < json['postprocessors']['output_streams'].length; ++i) {
      let node = json['postprocessors']['output_streams'][i]
      let input_streams = node['streams']
      for (let j =0; j < input_streams.length; ++j) {
        if (!valid_stream_types.includes(input_streams[j])) {
          alert("Invalid stream type in " + node.name + ": " + input_streams[j]);
          return false;
        }
      }
    }
    return true;
  }
}

$('#btn-preview-config').click(function() {
  $('#json-preview').html(stringify(controller.exportJSON(), {maxLength: 80}));
  $('#preview-dialog').dialog('open');
});

$('#btn-write-config').click(function() {
  controller.writeConfig();
});

$('#btn-load-config').click(function() {
  controller.loadConfig();
});

$('#btn-load-default-config').click(function() {
  controller.deleteConfig();
  controller.loadDefaultConfig();
});

$('#btn-load-demo-config').click(function() {
  controller.loadDemoConfig();
});

$('.btn-load-preset').click(function() {
  controller.restoreConfig($(this).html());
});

$('.btn-save-preset').click(function() {
  controller.storeConfig($(this).html());
});

$('#file-upload-config').change(function() {
  let configFile = $(this).prop('files')[0];
  let fileReader = new FileReader();
  fileReader.onload = function() {
    controller.applyConfig(JSON.parse(fileReader.result));
    controller.tunable = false;
  };
  fileReader.readAsText(configFile);
  $(this).val('');
});

$(document).keydown(function(event) {
  var e = event.originalEvent;
  if (e.which < 112 || e.which > 114) return;
  let configKey = String.fromCharCode(e.which - 112 + 65);  // 'A' - 'C'

  if (e.ctrlKey) {
    controller.storeConfig(configKey);
  } else {
    controller.restoreConfig(configKey);
  }

  event.preventDefault();
});

if (window.location.host.indexOf('localhost') < 0) {
  $('#btn-load-demo-config').hide();
}

$('#btn-download-config').click(function() {
  let dataStr = 'data:text/json;charset=utf-8,' +
      encodeURIComponent(stringify(controller.exportJSON(), {maxLength: 80}));
  $('#download-anchor')
      .attr({'href': dataStr, 'download': 'cast_audio.json'})[0]
      .click();
});

$.getJSON('/setup/audio/get_avail_processors', function(avail_processors) {
  $.getJSON('data/config.json', function(config) {
    controller = new AudioTuningController(config, avail_processors);
    controller.init();
    controller.loadConfig();
  });
});
