#! /bin/sh

# Initialize/mount/.. persistent settings partition

SETTINGS_MOUNT_PATH="/media/settings"
SETTINGS_VOLUME_NAME="nsdk-settings"

# Try 'settings' partition first (on s800), otherwise 'data' (others)
MTD_PART=`cat /proc/mtd | grep '"settings"'`
UBI_VOL="ubi1"

if [ -z "${MTD_PART}" ]; then
	MTD_PART=`cat /proc/mtd | grep '"data"'`
	UBI_VOL="ubi0"
fi

MTD_PART=`echo ${MTD_PART} | sed -e 's/^mtd\([0-9]*\):.*/\1/'`
SUBPAGESIZE=`cat /sys/class/mtd/mtd${MTD_PART}/subpagesize`

# Keep compatibility with Sitara targets: we used 2048 instead of real subpages (512)
# so to be able to attach existing settings partition we need to stick to 2048.
#
# We can't easily detect this (only the kernel driver prints a message with expected value)
# so for now just don't go below 2048.
if [ "${UBI_VOL}" = "ubi1" ]; then
	if [ ${SUBPAGESIZE} -lt 2048 ]; then
		SUBPAGESIZE=2048
	fi
fi;

# Check whether the settings partition is on a different UBI device than rootfs
settings_separate() {
	if [ "${UBI_VOL}" = "ubi1" ]; then
		return 0
	else
		return 1
	fi
}

# Check whether settings partition is mounted
settings_mounted() {
	if mount | grep "${SETTINGS_VOLUME_NAME}" > /dev/null; then
		return 0
	else
		return 1
	fi
}

# Mount settings partition. Volume must be attached
settings_mount() {
	mkdir -p "${SETTINGS_MOUNT_PATH}"
	mount -t ubifs "${UBI_VOL}:${SETTINGS_VOLUME_NAME}" "${SETTINGS_MOUNT_PATH}"

	if settings_mounted; then
		echo "Settings partition mounted"
	fi
}

# Unmount the settings partition
settings_unmount() {
	# kill all processes keeping setting partition busy
	fuser -km "${SETTINGS_MOUNT_PATH}"
	umount "${SETTINGS_MOUNT_PATH}"
	return $?
}

# Attach the settings volume
settings_attach() {
	# attach settings only if on separate UBI volumes
	if settings_separate; then
		echo "Attaching settings volume"
		ubiattach -m ${MTD_PART} -O ${SUBPAGESIZE}
	fi
}

# Detach the settings volume. Volume must be unmounted
settings_detach() {
	# detach settings only if on separate UBI volumes
	if settings_separate; then
		ubidetach -m ${MTD_PART}
		echo "Settings partition unmounted & detached"
	fi
}

# Create the settings volume. Volume must be unmounted and detached.
# Afterwards, settings volume is attached.
settings_create() {
	echo "Creating settings partition"
	if settings_separate; then
		settings_detach
		ubiformat /dev/mtd${MTD_PART} -y -O ${SUBPAGESIZE}
		settings_attach
	else
		ubirmvol /dev/${UBI_VOL} -N "${SETTINGS_VOLUME_NAME}"
	fi

	ubimkvol /dev/${UBI_VOL} -m -N "${SETTINGS_VOLUME_NAME}"
}

# Recreate the settings volume
settings_reset() {
	echo "Settings factory reset"
	if settings_mounted; then

		if [ -d ${SETTINGS_MOUNT_PATH}/log ]; then
			echo "Saving logs to /tmp"
			rm -rf /tmp/log_backup
			cp -r ${SETTINGS_MOUNT_PATH}/log /tmp/log_backup
		fi

		if settings_unmount; then
			echo "Settings partition unmounted"
		else
			echo "Settings partition can't be unmounted"
			return 1
		fi
	fi

	settings_create

	if [ -d /tmp/log_backup ]; then
		echo "Restoring logs from /tmp"
		settings_mount
		mv /tmp/log_backup ${SETTINGS_MOUNT_PATH}/log
		chmod 777 ${SETTINGS_MOUNT_PATH}/log
		chmod 666 ${SETTINGS_MOUNT_PATH}/log/*
		settings_unmount
	fi

	settings_detach
}

# Temporarily save paired BLE RC files to rootfs
ble_rc_save() {
	echo "Save BLE RC files"

	# First we need to ensure that setting partition is mounted
	# because BLE RC keys are stored there.
	if ! settings_mounted; then
		if [ -n "${MTD_PART}" ]; then
			settings_attach
			if ubinfo /dev/${UBI_VOL} -a | grep "${SETTINGS_VOLUME_NAME}" > /dev/null; then
				echo -e "Settings partition already initialized"
			else
				return 1
			fi

			settings_mount

			if ! settings_mounted; then
				return 1
			fi
		fi
	fi

	# all files that are needed are stored in
	# /media/settings/bluetooth/lib/bluetooth/"deviceMac"/"bleRcMac"/*
	root="/media/settings/bluetooth/lib/bluetooth"
	if [ -d ${root} ]; then
		device_mac=$(ls "$root"| grep -o "[[:xdigit:]:]\{11,17\}")
		destination="/usr/share/nsdk/ble-rc"
		if [ -n $device_mac ]; then
			# There can be more folders insede
			# /media/settings/bluetooth/lib/bluetooth/"deviceMac"/
			# One for every paired bt device so we need to find rc
			for folder in "$root/${device_mac}"/*; do
				if [ -d $folder ] && [ -f "${folder}/info" ]; then
					# There is parameter "Source" insede
					# /media/settings/bluetooth/lib/bluetooth/"deviceMac"/"bleRcMac"/info file
					# which specifies type of device. 1 - audio device, 2 - ble rc
					cat "${folder}/info" | grep 'Services=' | grep '00001812-0000-1000-8000-00805f9b34fb' &> /dev/null
					if [ $? == 0 ]; then
						# We create temporary folder /usr/share/nsdk/ble-rc/"deviceMac"
						# where we store folder with ble rc files
						if [ ! -d "${destination}/${device_mac}" ]; then
							mkdir -p "${destination}/${device_mac}"
						fi
						echo "Saving..."
						cp -r $folder "${destination}/${device_mac}"

						# We also save folder contain of cache which applies to ble rc
						rc_mac=$(basename $folder)
						rc_cache_file="${root}/${device_mac}/cache/${rc_mac}"
						if [ -f $rc_cache_file ]; then
							mkdir "${destination}/${device_mac}/cache"
							cp $rc_cache_file "${destination}/${device_mac}/cache/"
						fi

						echo "Done"
						return 0
					fi
				fi
			done
		fi
	fi
	echo "Nothing to be saved"
	return 1
}

# Restore paired BLE RC from rootfs to seeting partition
ble_rc_restore() {
	echo "Restore BLE RC files"

	# First we check if there are any ble rc files at all
	root="/usr/share/nsdk/ble-rc"
	if [ -d ${root} ]; then
		device_mac=$(ls "$root"| grep -o "[[:xdigit:]:]\{11,17\}")
		if [ -n $device_mac ]; then
			# If there are files to be restored we create appropraite folder inside
			# setting partition if it doesnt exist and copy files there
			# Folder /usr/share/nsdk/ble-rc is then erased so this happens only after fdr
			# and not after every boot
			destination="/media/settings/bluetooth/lib/bluetooth/${device_mac}"
			if [ ! -d $destination ]; then
				mkdir -p $destination
			fi
			echo "Restoring..."
			cp -r "${root}/${device_mac}"/* $destination
			rm -r $root
			echo "Done"
			return
		fi
	fi
	echo "Nothing to be restored"
}

if [ -e /usr/share/nsdk/factory_reset ]; then
	if [ -n "${MTD_PART}" ]; then
		# Before reseting setting partiton we save ble rc files so paired BLE RC is not lost
		ble_rc_save
		settings_reset
	else
		echo -e "ERROR: Settings partition not found; clearing ${SETTINGS_MOUNT_PATH}"
		rm -rf "${SETTINGS_MOUNT_PATH}"
	fi
	rm -f /usr/share/nsdk/factory_reset
fi

if settings_mounted; then
	echo "Settings partition already mounted"
else
	if [ -z "${MTD_PART}" ]; then
		echo -e "ERROR: Settings partition not found; using rootfs!"
	else
		settings_attach
		if ubinfo /dev/${UBI_VOL} -a | grep "${SETTINGS_VOLUME_NAME}" > /dev/null; then
			echo -e "Settings partition already initialized"
		else
			settings_create
		fi

		settings_mount

		# one-time recovery if mount failed for some reason
		if ! settings_mounted; then
			echo "ERROR mounting settings partition - re-create it"
			settings_create
			settings_mount

			if ! settings_mounted; then
				echo "ERROR: Unable to mount/recreate the settings partition!"
			fi
		fi
	fi

	# make sure settings symlinks exist and point to proper locations
	if [ ! -e "${SETTINGS_MOUNT_PATH}"/settings ]; then
		mkdir -p "${SETTINGS_MOUNT_PATH}"/settings
	fi
	if [ ! -h /usr/share/nsdk/settings ]; then
		ln -sf "${SETTINGS_MOUNT_PATH}"/settings /usr/share/nsdk/settings
	fi

	# make sure data-rw symlinks exist and point to proper locations
	if [ ! -e "${SETTINGS_MOUNT_PATH}"/data-rw ]; then
		mkdir -p "${SETTINGS_MOUNT_PATH}"/data-rw
	fi
	if [ ! -h /usr/share/nsdk/data-rw ]; then
		ln -sf "${SETTINGS_MOUNT_PATH}"/data-rw /usr/share/nsdk/data-rw
	fi

	# make settings and data-rw available for nsdk group
	chgrp nsdk "${SETTINGS_MOUNT_PATH}"
	chmod 0775 "${SETTINGS_MOUNT_PATH}"
	chgrp nsdk "${SETTINGS_MOUNT_PATH}"/data-rw
	chmod 0775 "${SETTINGS_MOUNT_PATH}"/data-rw
fi

# After setting partition is mounted we check if there are BLE RC files for restoration
# and restore them to setting partition af there are
ble_rc_restore
