#!/bin/sh
#
# Start playback of specified type of audio and record it with own microphones

set -e

. /usr/share/factory-tool/scripts/factory_tool_include

VOLUME="$2"

show_help()
{
cat << EOF
Usage:
	factory_tool test_audio_selfrecord sine <VOLUME> <DURATION> <FREQUENCY> [OUT_CHANNEL]
	factory_tool test_audio_selfrecord pink <VOLUME> <DURATION> [OUT_CHANNEL]
	factory_tool test_audio_selfrecord sweep <VOLUME> [OUT_CHANNEL]
	factory_tool test_audio_selfrecord aux <VOLUME> <DURATION> [OUT_CHANNEL]

Description:
	Start playback of corresponding type of audio and record it with
	device microphones. In case of aux option - aux will be used as source
	for playback.
	Recording will be saved in
	/tmp/sine_selfrecord.wav
	/tmp/pink_selfrecord.wav
	/tmp/sweep_selfrecord.wav
	/tmp/aux_selfrecord.wav
	respectively.

Parameters:
	VOLUME
		Volume level.
		Value should be between 0 and 1.
	DURATION
		Duration of specified audio (seconds).
	FREQUENCY
		Frequency of sine wave in Hz.
	OUT_CHANNEL
		Which output channel to use.
		Value should be between 1 and 99 or all (default).
		Value shouldn't exceed maximum channel count.
EOF
}

case "$1" in
	sine)
		if [ $# -ne 5 ] && [ $# -ne 4 ]; then
			echo "Error: wrong amount of parameters"
			show_help
			exit 1
		fi
		# Assign rest of parameters
		DURATION="$3"
		FREQ="$4"
		# Assign "all" value, if OUT_CHANNEL parameter is not provided
		if [ "$5" = "" ]; then
			OUT_CHANNEL="all"
		else
			OUT_CHANNEL="$5"
		fi

		check_channel "$OUT_CHANNEL" "$MAX_OUTPUT_CH"

		arecord -f S32_LE -r 48000 -c "$MAX_MICS_INPUT_CH" -t wav -D mics /tmp/sine_selfrecord.wav &

		MIX_MATRIX=`generate_channel_mix_matrix 1 "$MAX_OUTPUT_CH" 1 "$OUT_CHANNEL"`

		gst-launch-1.0 audiotestsrc freq=$FREQ is-live=true volume=$VOLUME ! \
			"audio/x-raw,format=S32LE,rate=48000,channels=1" ! \
			audioconvert mix-matrix="$MIX_MATRIX" ! \
			"audio/x-raw,format=S32LE,rate=48000,channels=$MAX_OUTPUT_CH" ! \
			autoaudiosink &
		sleep "$DURATION"
		# Kill background processes after sleep
		pkill -P $$
		;;

	pink)
		if [ $# -ne 4 ] && [ $# -ne 3 ]; then
			echo "Error: wrong amount of parameters"
			show_help
			exit 1
		fi
		# Assign rest of parameters
		DURATION="$3"
		# Assign "all" value, if OUT_CHANNEL parameter is not provided
		if [ "$4" = "" ]; then
			OUT_CHANNEL="all"
		else
			OUT_CHANNEL="$4"
		fi

		check_channel "$OUT_CHANNEL" "$MAX_OUTPUT_CH"

		arecord -f S32_LE -r 48000 -c "$MAX_MICS_INPUT_CH" -t wav -D mics /tmp/pink_selfrecord.wav &

		MIX_MATRIX=`generate_channel_mix_matrix 1 "$MAX_OUTPUT_CH" 1 "$OUT_CHANNEL"`

		gst-launch-1.0 audiotestsrc wave=6 is-live=true volume=$VOLUME ! \
			"audio/x-raw,format=S32LE,rate=48000,channels=1" ! \
			audioconvert mix-matrix="$MIX_MATRIX" ! \
			"audio/x-raw,format=S32LE,rate=48000,channels=$MAX_OUTPUT_CH" ! \
			autoaudiosink &
		sleep "$DURATION"
		# Kill background processes after sleep
		pkill -P $$
		;;

	sweep)
		if [ $# -ne 3 ] && [ $# -ne 2 ]; then
			echo "Error: wrong amount of parameters"
			show_help
			exit 1
		fi
		# Assign "all" value, if OUT_CHANNEL parameter is not provided
		if [ "$3" = "" ]; then
			OUT_CHANNEL="all"
		else
			OUT_CHANNEL="$3"
		fi

		check_channel "$OUT_CHANNEL" "$MAX_OUTPUT_CH"

		arecord -f S32_LE -r 48000 -c "$MAX_MICS_INPUT_CH" -t wav -D mics /tmp/sweep_selfrecord.wav &

		MIX_MATRIX=`generate_channel_mix_matrix 1 "$MAX_OUTPUT_CH" 1 "$OUT_CHANNEL"`

		gst-launch-1.0 filesrc location=/usr/share/factory-tool/files/sweep-20-22k-10s.ogg ! \
			oggdemux ! vorbisdec ! audioconvert ! "audio/x-raw,format=S32LE,rate=48000,channels=1" ! \
			audioconvert mix-matrix="$MIX_MATRIX" ! "audio/x-raw,format=S32LE,rate=48000,channels=$MAX_OUTPUT_CH" ! \
			volume volume=$VOLUME ! audioresample ! autoaudiosink
		pkill -P $$
		;;

	aux)
		if [ $# -ne 4 ] && [ $# -ne 3 ]; then
			echo "Error: wrong amount of parameters"
			show_help
			exit 1
		fi
		# Assign rest of parameters
		DURATION="$3"
		# Assign "all" value, if OUT_CHANNEL parameter is not provided
		if [ "$4" = "" ]; then
			OUT_CHANNEL="all"
		else
			OUT_CHANNEL="$4"
		fi

		check_channel "$OUT_CHANNEL" "$MAX_OUTPUT_CH"

		arecord -f S32_LE -r 48000 -c "$MAX_MICS_INPUT_CH" -t wav -D mics /tmp/aux_selfrecord.wav &

		MIX_MATRIX=`generate_channel_mix_matrix "$MAX_AUX_INPUT_CH" "$MAX_OUTPUT_CH" all "$OUT_CHANNEL"`

		gst-launch-1.0 alsasrc device=aux ! \
			"audio/x-raw,format=S32LE,rate=48000,channels=$MAX_AUX_INPUT_CH" ! \
			audioconvert mix-matrix="$MIX_MATRIX" ! \
			"audio/x-raw,format=S32LE,rate=48000,channels=$MAX_OUTPUT_CH" ! \
			volume volume=$VOLUME ! autoaudiosink &
		sleep "$DURATION"
		pkill -P $$
		;;

	*)
		show_help
		;;
esac

exit 0
