"use strict";

var NSDK = (function($) {
	var root_uri = "/api";
	var event_uri = "/api/event";

	/* low-level API functions */

	function _getData(path, roles) {
		var ret = $.Deferred();
		var uri = root_uri + "/getData";

		$.getJSON(uri, {
				'path': path,
				'roles': roles,
				'_nocache': _nocache()
			})
			.always(function(data) {
				console.log("getData on " + path + " finished");
			})
			.done(function(data) {
				if (data.status >= 400) {
					console.log(data.responseText);
					ret.reject(data.responseText);
				} else {
					if (!$.isEmptyObject(data.error)) {
						console.log("getData error: ", data.error);
						console.log(data.error);
						ret.reject(data);
					} else {
						console.log("getData response: ", data);
						console.log(data);
						ret.resolve(data);
					}
				}
			})
			.fail(function(jqXHR) {
				// StreamSdk returns HTTP 500 if the request was answered with an error at the application
				// layer, this code is also called in case there was a problem in the communication
				// (timeout, etc.).
				console.error("Get data failed!", path, roles);
				try {
					var responseObj = jQuery.parseJSON(jqXHR.responseText);
					console.error("GetData error: " + responseObj.error.name + ": " + responseObj.error.message + " (" + jqXHR.statusText + ")");
					ret.reject(responseObj);
				} catch (e) {
					console.error("Failed to parse GetData error: " + e.message);
					ret.reject(null);
				}
			});

		return ret.promise();
	}

	function _setData(path, role, value, success) {
		var ret = $.Deferred();
		var uri = root_uri + "/setData";

		$.getJSON(uri, {
				'path': path,
				'role': role,
				'value': JSON.stringify(value, null),
				'_nocache': _nocache()
			})
			.always(function() {
				console.log("setData on " + path + " finished.");
			})
			.done(function(data) {
				console.log("setData response: ", data);
				ret.resolve(data);
			})
			.fail(function(jqXHR) {
				// StreamSdk returns HTTP 500 if the request was answered with an error at the application layer, this code is also
				// called in case there was a problem in the communication (timeout, etc.).
				try {
					var responseObj = jQuery.parseJSON(jqXHR.responseText);
					console.error("setData error: " + responseObj.error.name + ": " + responseObj.error.message + " (" + jqXHR.statusText + ")");
					ret.reject(responseObj);
				} catch (e) {
					console.error("Failed to parse SetData error: " + e.message);
					ret.reject(null);
				}
			});

		return ret.promise();
	}

	function _getRows(path, roles, from, to) {
		var ret = $.Deferred();
		var uri = root_uri + "/getRows";

		$.getJSON(uri, {
				'path': path,
				'roles': roles,
				'from': from.toString(),
				'to': to.toString(),
				'_nocache': _nocache()
			})
			.always(function(data) {
				console.log("getRows on " + path + " finished (from=" + from.toString() + ", to=" + to.toString() + ")");
			})
			.done(function(data) {
				// console.log("getRows response: ", data);
				ret.resolve(data);
			})
			.fail(function(jqXHR) {
				try {
					var responseObj = jQuery.parseJSON(jqXHR.responseText);
					console.error("getRows error: " + responseObj.error.name + ": " + responseObj.error.message + " (" + jqXHR.statusText + ")");
					responseObj.rowsCount = 0;
					responseObj.rowsVersion = 0;
					responseObj.rowsRedirect = 0;
					responseObj.rowsRoles = [];
					responseObj.rows = [];
					ret.reject(responseObj);
				} catch (e) {
					console.error("Failed to parse GetRows error: " + e.message);
					ret.reject(null);
				}
			});

		return ret.promise();
	}

	/* Low-level event functions */
	function _modifyQueue(queueId, subscribe, unsubscribe, success) {
		var uri = event_uri + "/modifyQueue";

		return $.getJSON(uri, {
			'queueId': queueId,
			'subscribe': JSON.stringify(subscribe),
			'unsubscribe': JSON.stringify(unsubscribe),
			'_nocache': _nocache()
		}, success);
	}

	function _pollQueue(queueId, timeout, success) {
		var uri = event_uri + "/pollQueue";

		return $.getJSON(uri, {
			'queueId': queueId,
			'timeout': timeout,
			'_nocache': _nocache()
		}, success);
	}

	/* Higher-level requests with cleaned-up data */
	function _rowToDict(row, roles) {
		var ret = {};
		for (var role in roles) {
			ret[roles[role]] = row[role];
		};

		return ret;
	}

	/* Private Functions */
	function _nocache() {
		// #milliseconds since 1970/01/01 to prevent Internet Explorer caching
		return new Date().getTime().toString();
	}

	return {
		getData: _getData,
		setData: _setData,
		getRows: _getRows,
		modifyQueue: _modifyQueue,
		pollQueue: _pollQueue,
		rowsToDicts: function(rows, roles) {
			var ret = [];
			for (var row in rows) {
				ret[row] = _rowToDict(rows[row], roles);
			};
			return ret;
		}
	};
})(jQuery);

function getTypedValue(value) {
	switch (value.type) {
		case "string_":
			return value.string_;
		case "i32_":
			return value.i32_;
		case "i64_":
			return value.i64_;
		case "bool_":
			return value.bool_;
		case "double_":
			return value.double_;
		case "doubleList":
			return value.doubleList;
		default:
			return JSON.stringify(value[value.type]);
	}
}

function stringToValue(str, type) {
	var v;
	switch (type) {
		case 'string_':
			v = str.toString();
			break;
		case 'i32_':
			v = parseInt(str);
			break;
		case 'i64_':
			v = parseInt(str);
			break;
		case 'bool_':
			if (str == 'true') v = true;
			else if (str == 'false') v = false;
			else window.alert("Unknown bool value " + str);
			break;
		case 'double_':
			v = parseFloat(str);
			break;
		default:
			window.alert("Unknown type " + type);
			break;
	}

	var ret = {
		type: type
	};
	ret[type] = v;
	return ret;
}

/******************************************************************************/
/* NSDK API Helper */

function NSDK_GetTypedValue(nsdk_value) {
	// not objected value, return directly
	if (typeof(nsdk_value) !== 'object') {
		return nsdk_value;
	}

	// parse objected nsdk value
	return getTypedValue(nsdk_value);
}

function NSDK_GetData(path, roles) {
	var ret = jQuery.Deferred();

	NSDK.getData(path, roles)
		.done(function(data) {
			// data are expected to be an array
			if ($.isArray(data)) {
				var result = {};
				roles = roles.split(","); // connvert , separeted roles to array
				// make each num indexed role also associative indexed
				for (var i = 0; i < roles.length; i++) {
					result[$.trim(roles[i])] = data[i];
					result[i] = data[i];
				}
				console.log("got data '" + path + "': ", result);
				ret.resolve(result);
			} else {
				var error = {
					message: "Not the array returned for '" + path + "' but: " + data.toString()
				};
				console.error(error.message);
				ret.reject(error);
			}
		})
		.fail(function() {
			var error = {
				message: "Error to get data " + path
			};
			console.error(error.message);
			ret.reject(error);
		});

	return ret.promise();
};

function NSDK_GetRows(path, roles, from, to) {
	var ret = jQuery.Deferred();

	NSDK.getRows(path, roles, from, to)
		.done(function(data) {
			// data must be an object with rowsCount attribute
			if ((typeof(data) === 'object') && (typeof(data.rowsCount) !== 'undefined')) {
				roles = roles.split(","); // connvert , separeted roles to array
				// make each num indexed role also associative indexed
				for (var j = 0; j < data.rows.length; j++) {
					var row = {};
					for (var i = 0; i < roles.length; i++) {
						row[$.trim(roles[i])] = data.rows[j][i];
						row[i] = data.rows[j][i];
					}
					data.rows[j] = row;
				}
				console.log("got rows '" + path + "' count: " + data.rowsCount + " rows: ", data.rows);
				ret.resolve(data);
			} else {
				var error = {
					message: "Not the object with rowsCount returned for '" + path + "' but: " + data.toString()
				};
				console.error(error.message);
				ret.reject(error);
			}
		})
		.fail(function() {
			var error = {
				message: "Error to get rows " + path
			};
			console.error(error.message);
			ret.reject(error);
		});

	return ret.promise();
}

function NSDK_SetData(path, data, role_optional) {
	var role;
	if (typeof(role_optional) === 'undefined') {
		role = "value";
	} else {
		role = role_optional;
	}

	var nsdk_value;
	switch (typeof(data)) {
		case 'boolean':
			nsdk_value = stringToValue(data.toString(), "bool_");
			break;
		case 'string':
			nsdk_value = stringToValue(data, "string_");
			break;
		case 'number':
			nsdk_value = stringToValue(data, "i32_");
			break;
		default:
		case 'object':
			nsdk_value = data;
			break;
	}

	return NSDK.setData(path, role, nsdk_value);
}

function NSDK_Activate(path, _optional /* defaults to: {} */ ) {
	var value;
	if (typeof(_optional) === 'undefined') {
		value = {};
	} else {
		value = _optional;
	}

	return NSDK_SetData(path, value, "activate");
}

function NSDK_ActivateArrayResult() {
	this.done_count = 0;
	this.fail_count = 0;
};

function NSDK_ActivateArray(paths_array, cb_onResult__result) {
	var result = new NSDK_ActivateArrayResult();

	if (paths_array.length == 0) {
		cb_onResult__result(result);
		return;
	}

	function activateItems(fragment) {
		if (fragment.length > 0) {
			var item = fragment.shift();
			NSDK_Activate(item)
				.done(function() {
					result.done_count++;
					activateItems(fragment);
				})
				.fail(function() {
					result.fail_count++;
					activateItems(fragment);
				});
		} else {
			cb_onResult__result(result);
		}
	}

	activateItems(paths_array.slice(0));
}

/******************************************************************************/
/* NSDK API further helper */

/**
 * Resolves value role of given node decoded by NSDK_GetTypedValue
 * @param {string} path NSDK path to node which have a value role
 * @returns {jQuery.Deferred}
 */
function NSDK_ReadValue(path) {
	var ret = jQuery.Deferred();

	NSDK_GetData(path, "value").always(function(response) {
		if (response && response["value"]) {
			ret.resolve(NSDK_GetTypedValue(response["value"]));
		} else {
			ret.reject();
		}
	});

	return ret.promise();
}

/**
 * Store NSDK's stringToValue encoded value into value role of given node
 * @param {string} path NSDK path to node which have a value role
 * @param {boolean|string|object} value (type sypported by stringToValue)
 * @returns {jQuery.Deferred}
 */
function NSDK_WriteValue(path, value) {
	var ret = jQuery.Deferred();

	NSDK_SetData(path, value, "value").done(function() {
		ret.resolve(true);
	}).fail(function() {
		ret.reject(false);
	});

	return ret.promise();
}

/*
 * Performs NSDK_WriteValue and NSDK_ReadValue consecutively, usefull to verify the written value by user code
 * @param {string} NSDK path to node which have a value role
 * @param {boolean|string|object} value (type sypported by stringToValue)
 * @returns {jQuery.Deferred}
 */
function NSDK_WriteReadValue(path, value) {
	var ret = jQuery.Deferred();

	NSDK_SetData(path, value, "value").always(function() {
		NSDK_GetData(path, "value").always(function(response) {
			if (response && response["value"]) {
				ret.resolve(NSDK_GetTypedValue(response["value"]));
			} else {
				ret.reject();
			}
		});
	});

	return ret.promise();
}

/**
 * Reads object typed node
 * @param {string} path
 * @param {string} type
 * @param {function} cb__type function to pass result of type: type
 */
function NSDK_ReadObject(path, type, cb__type) {
	NSDK_GetData(path, "value").always(function(response) {
		if (response &&
			response.value &&
			response.value.type &&
			response.value.type === type &&
			response.value[type]
		) {
			cb__type(response.value[type]);
		} else {
			cb__type( /*undefined*/ );
		}
	});
}

/**
 * Writes object typed node
 * @param {string} path
 * @param {object} object
 * @param {string} type
 * @param {function|undefined} cb__bool callback with bool parameter to process succeed|failed result
 */
function NSDK_WriteObject(path, object, type, cb__bool) {
	var value = {};
	value.type = type;
	value[type] = object;

	NSDK_SetData(path, value, "value").done(function() {
		if (cb__bool) {
			cb__bool(true);
		}
	}).fail(function() {
		if (cb__bool) {
			cb__bool(false);
		}
	});
}

/**
 * Singleton class to subscribe to nsdk events,
 * Types to subscribe:
 *  - item - notified everytime data on path are changed
 *  - itemWithValue - notified everytime value on path is changes
 *  - rows - notified when rows on path are changed
 */
var NSDK_Subscribe = (function($) {
	var eventHandler = {};

	/* Event queue handling */
	var queueId = "";
	var resubscribe = false;

	/**
	 * One iteration in queue polling cycle
	 */
	function _queueIteration() {
		var ret = null;
		if ((queueId == "") || resubscribe == true)
			ret = _renewQueue();
		else
			ret = _pollQueue();

		ret.fail(function() {
			queueId = "";
			setTimeout(_queueIteration, 1000);
		});
		ret.done(function() {
			setTimeout(_queueIteration)
		});
		return ret;
	}

	/**
	 * Renews event queue on the remote side.
	 */
	function _renewQueue() {
		resubscribe = false;
		var subscribes = [];

		for (var path in eventHandler) {
			for (var type in eventHandler[path]) {
				subscribes.push({
					'path': path,
					'type': type
				});
			}
		}
		return NSDK.modifyQueue(queueId, subscribes, [])
			.done(function(data) {
				/* Queue was successfully renewed after loss. Announce changes
				 * on all subscribed paths to ensure path is up-to-date.
				 */
				queueId = data;
			});
	}

	/**
	 * Issues poll request to a queue and if succeeds,call s
	 * _handleEvents to process received events. If there was a
	 * communication error, reset queueId
	 */
	function _pollQueue() {
		return NSDK.pollQueue(queueId, 1500)
			.done(function(data) {
				_handleEvents(data);
			})
			.fail(_queueFailed);
	}

	function _subscribe(path, type, handler) {
		console.log("Subscribe path: " + path + " type: " + type);
		var added = false;
		if (handler !== undefined && handler !== null) {
			if (eventHandler[path] === undefined) {
				/* This path does not exist */
				eventHandler[path] = [];
				eventHandler[path][type] = [];
				eventHandler[path][type].push(handler);
				added = true;
			} else if (eventHandler[path][type] === undefined) {
				/* This path already exist, but with other type*/
				eventHandler[path][type] = [];
				eventHandler[path][type].push(handler);
				added = true;
			} else {
				/* This path with given type already exists. Add handler but do not modify queue*/
				console.log("This path with given type already exists. Path: " + path + ", type: " + type + ", Adding handler.");
				eventHandler[path][type].push(handler);
			}
		} else {
			console.error("Subscribe path: " + path + " type: " + type + " doesn't provide handler!");
		}

		if (queueId == "") {
			resubscribe = true;
		} else if (added) {
			NSDK.modifyQueue(queueId, [{
					'path': path,
					'type': type
				}], [])
				.fail(_queueFailed);
		}

		return {
			"path": path,
			"type": type,
			"handler": handler
		}
	}

	function _handleEvents(events) {
		var path = "";

		for (var i = 0; i < events.length; ++i) {
			path = events[i].path;

			// copy of available handlers for the path, so handle
			// method can call unsubscribe without breaking iteration
			var handlers = [];

			for (var type in eventHandler[path]) {
				for (var handler = 0; handler < eventHandler[path][type].length; handler++) {
					handlers.push(eventHandler[path][type][handler]);
				}
			}

			for (var handler = 0; handler < handlers.length; handler++) {
				handlers[handler](events[i]);
			}
		}
	}

	function _queueFailed() {
		queueId = "";
	}

	function _unsubscribe(handle) {
		if (handle !== undefined && handle !== null) {
			console.log("Unsubscribe path: " + handle.path + " type: " + handle.type);
			var path = handle.path;
			var type = handle.type;
			var removed = false;

			if (eventHandler[path] !== undefined && eventHandler[path][type] !== undefined) {
				for (var i = 0; i < eventHandler[path][type].length; i++) {
					if (eventHandler[path][type][i] === handle.handler) {
						eventHandler[path][type].splice(i, 1);
						break;
					}
				}
				if (eventHandler[path][type].length == 0) {
					// last path of type 'type' was removed,
					// remove also from queue
					removed = true;
					delete eventHandler[path][type];
				}

				if (Object.keys(eventHandler[path]).length == 0) {
					delete eventHandler[path];
				}
			}

			if (queueId == "") {
				resubscribe = true;
			} else if (removed) {
				NSDK.modifyQueue(queueId, [], [{
					'path': path,
					'type': type
				}]).fail(_queueFailed);
			}
		}
	}

	// Start queue polling
	_queueIteration();

	return {
		subscribe: _subscribe,
		unsubscribe: _unsubscribe
	};
})(jQuery);
