class Utils {
  static asciiFromBuffer(buffer, range) {
    return String.fromCharCode
      .apply(null, buffer.slice(range.min, range.max))
      .trim()
      .replace(/\0/g, "");
    }

  static parseIPAddress(buffer) {
    return buffer.join(".");
  }

  static parseSignedValue(value) {
    if (value > 128) {
      value = -value + 128;
    }
    return value;
  }

  static parseVolumeLevel(value) {
    if (value > 128) {
      return -((value - 0x80) * 0.5);
    }

    return value * 0.5;
  }

  static encodeVolumeLevel(value) {
    return this.toSignedValue(value * 2);
  }

  static parseMACAddress(buffer) {
    return [...buffer].map(hexToPaddedString).join(":");
  }

  static parseASCII(buffer) {
    return buffer
    .toString("utf8")
    .trim()
    .replace(/\0/g, "");
  }

  static asciiToHex(ascii) {
    return ascii
    .split("")
    .map(x => parseInt(x.charCodeAt(0)) );
  }

  static toSignedValue(input) {
    if (input < 0) {
      input = Math.abs(input) + 128;
    }
    return input;
  }

  static toSubStereoValue(input) {
    if (input === 0) {
      return input;
    }
    return Math.abs(input) * 2 + 128;
  }

  static stringToPaddedHex(input, count) {
    var output = [];
    for (var i = 0; i < count; i++) {
      var h = Number(input.charCodeAt(i));
      output.push(isNaN(h) ? 0 : h);
    }
    return output;
  }

  static audioFormat(encoding, channel) {
    if (encoding === 0x15 && (channel === 0x21 || channel === 0x2A)) {
      return "---";
    }

    // Atmos, DTS:X, IMAX
    if (encoding === 0x16 || encoding === 0x17 || encoding === 0x18) {
      return this.parseEncoding(encoding);
    }

    // auro
    if (channel >= 0x30 && channel <= 0x38) {
      return this.parseChannelConfig(channel);
    }

    return this.parseEncoding(encoding) + " " + this.parseChannelConfig(channel);
  }

  static parseEncoding(value) {
    switch (value) {
      case 0x00:
      return "PCM";
      case 0x01:
      return "Analogue Direct";
      case 0x02:
      return "Dolby Digital";
      case 0x03:
      return "Dolby Digital EX";
      case 0x04:
      return "Dolby Digital Surround";
      case 0x05:
      return "Dolby Digital Plus";
      case 0x06:
      return "Dolby Digital True HD";
      case 0x07:
      return "DTS";
      case 0x08:
      return "DTS 96/24";
      case 0x09:
      return "DTS ES Matrix";
      case 0x0a:
      return "DTS ES Discrete";
      case 0x0b:
      return "DTS ES Matrix 96/24";
      case 0x0c:
      return "DTS ES Discrete 96/24";
      case 0x0d:
      return "DTS HD Master Audio";
      case 0x0e:
      return "DTS HD High Res Audio";
      case 0x0f:
      return "DTS Low Bit Rate";
      case 0x10:
      return "DTS Core";
      case 0x13:
      return "PCM Zero";
      case 0x14:
      return "Unsupported";
      case 0x15:
      return "Undetected";
      case 0x16:
      return "Dolby Atmos";
      case 0x17:
      return "DTS:X";
      case 0x18:
      return "IMAX ENHANCED";
      case 0x19:
      return "Auro 3D";
    }
  }

  static parseChannelConfig(value) {
    switch (value) {
      case 0x00:
      case 0x01:
      return "1.0";
      case 0x02:
      case 0x0e:
      case 0x0f:
      return "2.0";
      case 0x03:
      return "2/1.0";
      case 0x04:
      return "2/2.0";
      case 0x05:
      return "2/3.0";
      case 0x06:
      return "2/4.0";
      case 0x07:
      return "2/2.0 + matrix";
      case 0x08:
      return "3.0";
      case 0x09:
      return "3/1.0";
      case 0x0a:
      return "3/2.0";
      case 0x0b:
      return "3/3.1";
      case 0x0c:
      return "3/4.0";
      case 0x0d:
      return "3/2.0 + matrix";
      case 0x10:
      case 0x11:
      return "1.1";
      case 0x12:
      case 0x1e:
      return "2.1";
      case 0x13:
      return "2/1.1";
      case 0x14:
      case 0x17:
      return "2/2.1";
      case 0x15:
      return "2/3.1";
      case 0x16:
      return "2/4.1";
      case 0x18:
      return "3/0.1 + matrix";
      case 0x19:
      return "3/1.1";
      case 0x1a:
      return "3/2.1";
      case 0x1b:
      return "3/3.1";
      case 0x1c:
      return "3/4.1";
      case 0x1d:
      return "3/2.1 + matrix";
      case 0x21:
      return ""; // Undetected
      case 0x30:
      return "Auro Quad";
      case 0x31:
      return "Auro 5.0";
      case 0x32:
      return "Auro 5.1";
      case 0x33:
      return "Auro 2.2.2";
      case 0x34:
      return "Auro 8.0";
      case 0x35:
      return "Auro 9.1";
      case 0x36:
      return "Auro 10.1";
      case 0x37:
      return "Auro 11.1";
      case 0x38:
      return "Auro 13.1";
      default:
      return "Unknown";
    }
  }

  static parseSampleRate(value) {
    switch (value) {
      case 0x00:
      return "32 KHz";
      case 0x01:
      return "44.1 KHz";
      case 0x02:
      return "48 KHz";
      case 0x03:
      return "88.2 KHz";
      case 0x04:
      return "96 KHz";
      case 0x05:
      return "176.4 KHz";
      case 0x06:
      return "192 KHz";
      case 0x07:
      return "Unknown";
      case 0x08:
      return "---";
    }
  }

  static parseBitrate(value) {
    switch (value) {
      case 0x00:
      return "32kbps";
      case 0x01:
      return "56kbps";
      case 0x02:
      return "64kbps";
      case 0x03:
      return "96kbps";
      case 0x04:
      return "112kbps";
      case 0x05:
      return "128kbps";
      case 0x06:
      return "192kbps";
      case 0x07:
      return "224kbps";
      case 0x08:
      return "256kbps";
      case 0x09:
      return "320kbps";
      case 0x0a:
      return "384kbps";
      case 0x0b:
      return "448kbps";
      case 0x0c:
      return "512kbps";
      case 0x0d:
      return "576kbps";
      case 0x0e:
      return "640kbps";
      case 0x0f:
      return "768kbps";
      case 0x10:
      return "960kbps";
      case 0x11:
      return "1024kbps";
      case 0x12:
      return "1152kbps";
      case 0x13:
      return "1280kbps";
      case 0x14:
      return "1344kbps";
      case 0x15:
      return "1408kbps";
      case 0x16:
      return "1411.2kbps";
      case 0x17:
      return "1472kbps";
      case 0x18:
      return "1536kbps";
      case 0x19:
      return "1920kbps";
      case 0x1a:
      return "2048kbps";
      case 0x1b:
      return "3072kbps";
      case 0x1c:
      return "3840kbps";
      case 0x1d:
      return "Open";
      case 0x1e:
      return "Variable";
      case 0x1f:
      return "Lossless";
      default:
      return "---";
    }
  }

  static parseStereoDecodeMode(mode) {
    switch (mode) {
      case 0x01: return "Stereo";
      case 0x04: return "Dolby Surround";
      case 0x07: return "Neo:6 Cinema";
      case 0x08: return "Neo:6 Music";
      case 0x09: return "5/7 Ch Stereo";
      case 0x0A: return "DTS Neural:X";
      case 0x0B: return "Logic 16 Immersion";
      case 0x0C: return "DTS Virtual:X";
      case 0x0D: return "Dolby Virtual Height";
      case 0x0E: return "Auro Native";
      case 0x0F: return "Auro-Matic 3D";
      case 0x10: return "Auro-2D";
      default: return "---";
    }
 }

  static parseMultiChannelDecodeMode(mode) {
    switch (mode) {
      case 0x01: return "Stereo down-mix";
      case 0x02: return "Multi-channel";
      case 0x03: return "DTS Neural:X";
      case 0x06: return "Dolby Surround";
      case 0x0B: return "Logic 16 Immersion";
      case 0x0C: return "DTS Virtual:X";
      case 0x0D: return "Dolby Virtual Height";
      case 0x0E: return "Auro Native";
      case 0x0F: return "Auro-Matic 3D";
      case 0x10: return "Auro-2D";
      default: return "---";
    }
 }

  static parseDialNorm(value) {
    if (value == 255) {
      return "---";
    }
    return value.toString() + "dB";
  }

  static parseIncomingVideoResolution(data) {
    let msb = data[0];
    let lsb = data[1];
    let res = ((msb << 8) | lsb).toString();

    if (res === "0") {
      return "---";
    }

    let interlaced = data[3] === 1 ? "i" : "p";
    let fps = data[2].toString();

    return res + interlaced + fps;
  }
}

function hexToPaddedString(input) {
  let hex = input.toString(16);
  if (hex.length % 2) {
    hex = "0" + hex;
  }
  return hex.toUpperCase();
}

function bufferToHex(buffer) {
    return Array
        .from(new Uint8Array(buffer))
        .map(b => b.toString(16).padStart(2, "0"))
        .join(" ");
}

function chunkArray(array, chunkSize) {
    var result = [];

    for (var i = 0; i < array.length; i += chunkSize) {
      var chunk = array.slice(i, i + chunkSize);
      result.push(chunk);
    }

    return result;
}
