"use strict";

function Browser(browsing_root, table_view_selector, player_controller) {
	var self = this;
	this.page_stack = [];
	var root = browsing_root;
	var tableViewSelector = table_view_selector;
	var playerController = player_controller;

	var error = 0;
	var onErrorMessageHandlers = [];
	var onWorkingStateChangeHandlers = [];
	var onRenderRowsEventHandlers = [];

	this.getRoot = function() {
		return root;
	};

	this.getTableViewSelector = function() {
		return tableViewSelector;
	};

	this.getPlayerController = function() {
		return playerController;
	};

	/**
	 * Opens new browser page at given location and appends it to stack. Closes
	 * any menus on the stack.
	 *
	 * @param path the location to be opened
	 * @param context context data for menus
	 */
	this.openPage = function(path, context) {
		self.onWorkingStateChange(true);
		var numberOfPages = self.page_stack.length;
		var newPage = undefined;
		if (numberOfPages < 1 || (numberOfPages > 0 && self.page_stack[numberOfPages - 1].getPath() !== path)) {
			newPage = new BrowserPage(self, path, context);
			if (numberOfPages > 0) {
				self.deactivateCurrent();
			}
			self.page_stack.push(newPage);
		} else {
			newPage = self.page_stack[numberOfPages - 1];
		}

		NSDK_PROXY.browse(path)
			.done(function(data) {
				newPage.onBrowseFinished(data);
			})
			.fail(function(data) {
				self.onErrorMessage(data.error);
				self.closePage();
				self.onWorkingStateChange(false);
			});
	};

	/**
	 * Closes currently active browser page and removes it from the stack.
	 * Automatically closes any context pages that were also open.
	 */
	this.closePage = function() {
		var page;
		if (self.page_stack.length > 1) {
			self.onWorkingStateChange(true);
			page = self.page_stack.pop();
			page.destroy();
			self.closeAutoclosePages();
			page = self.page_stack[self.page_stack.length - 1];
			page.activate();
		}
	};

	/**
	 * Closes any context pages that are on the top of the stack.
	 */
	this.closeAutoclosePages = function() {
		var page;
		var nsdk_path;
		while (self.page_stack.length > 0) {
			page = self.page_stack[self.page_stack.length - 1];
			nsdk_path = NSDK_PROXY.pageAt(page.getPath());
			if (nsdk_path.get("containerType") == "context") {
				self.page_stack.pop();
				page.destroy();
			} else {
				return;
			}
		}
	};

	/**
	 * Closes currently active menu (all pages with context same as active one)
	 */

	this.closeMenu = function() {
		if (self.page_stack.length < 1)
			return;
		var top = self.page_stack[self.page_stack.length - 1];
		var menu_relatedRoles = top.getContext().relatedRoles;

		if (menu_relatedRoles == null)
			return;

		while ((top.getContext().relatedRoles || null) == menu_relatedRoles) {
			top.deactivate();
			if (self.page_stack.length > 0) {
				top.destroy();
				self.page_stack.pop();
				top = self.page_stack[self.page_stack.length - 1];
			}
		}
		top.activate();
	};

	this.closeContext = function(redirect) {
		if (self.page_stack.length < 1)
			return;
		var index;
		for (var t = 0; t < self.page_stack.length; ++t) {
			if (self.page_stack[t].getPath() == redirect) {
				index = t;
				break;
			}
		}
		if (index !== undefined) {
			var top = self.page_stack[self.page_stack.length - 1];
			top.deactivate();
			for (t = index; t < self.page_stack.length; ++t) {
				NSDK_PROXY.close(self.page_stack[t].getPath());
			}
			self.page_stack = self.page_stack.slice(0, index);
			top = self.page_stack[self.page_stack.length - 1];
		}
	};

	/**
	 *  Performs initialization of currently opened page
	 *
	 *  Activation usually consists of updating folder contents, re-generating
	 *  HTML code etc.
	 */
	this.activateCurrent = function() {
		if (self.page_stack.length > 0) {
			self.page_stack[self.page_stack.length - 1].activate();
		}
	};

	/**
	 * Performs cleanup actions if page is going the deactivated
	 */
	this.deactivateCurrent = function() {
		if (self.page_stack.length > 0) {
			var top_page = self.page_stack[self.page_stack.length - 1];
			top_page.deactivate();
		}
	};

	/**
	 * Destroys currently opened page
	 */
	this.destroyCurrent = function() {
		if (self.page_stack.length > 0) {
			var top_page = self.page_stack[self.page_stack.length - 1];
			top_page.deactivate();
			top_page.destroy();
		}
	};

	/**
	 * Deactivates  browser, closes active menus
	 */
	this.deactivate = function() {
		self.closeMenu();
	};

	this.activate = function() {};

	this.registerErrorMessageHandlers = function(h) {
		onErrorMessageHandlers.push(h);
		h(self.error);
	};

	this.registerWorkingStateChangeHandler = function(h) {
		onWorkingStateChangeHandlers.push(h);
	};

	this.registerRenderRowsEventHandler = function(h) {
		onRenderRowsEventHandlers.push(h);
	};

	this.onErrorMessage = function(error) {
		self.error = error;

		for (var i in onErrorMessageHandlers)
			onErrorMessageHandlers[i](self.error);
	};

	this.onWorkingStateChange = function(working) {
		for (var i in onWorkingStateChangeHandlers)
			onWorkingStateChangeHandlers[i](working);
	}

	this.onRenderRowsEvent = function(page) {
		for (var i in onRenderRowsEventHandlers)
			onRenderRowsEventHandlers[i](page);
	}

	$(tableViewSelector).tableview("registerActiveItemChangedHandler", function(data) {
		if (self.page_stack.length > 0)
			self.page_stack[self.page_stack.length - 1].fireEvent(data.id, data.context);
	});

	this.onTableViewPerPageChanged = function() {
		if (self.page_stack.length > 0) {
			var top_page = self.page_stack[self.page_stack.length - 1];
			top_page.renderContentTable(top_page.current_from);
		}
	};

	$(window).trigger("resize");

	// Open initial page
	self.openPage(this.getRoot(), {
		'roles': null
	});
}


// TODO: handle browser resize

/**
 * Creates a new BrowserPage object
 *
 * @param my_browser reference to parent browser object
 * @param my_path path displayed on this page
 * @param context optional context (used e.g. for carrying menu data)
 */
function BrowserPage(my_browser, my_path, context) {
	var self = this;
	var path = my_path;
	var browser = my_browser;
	var rows = [];
	var context = context;
	var disable_refresh = false;
	var page_error = 0;
	this.current_from = 0;
	var requested_path = my_path;

	var dummyRow = JSON.parse('{"type":"dummy","path":""}');

	if (context == undefined) context = null;

	// Dereference path
	//if (browser) browser.browser_api.getPath(my_path, function(success) {self.path = path; });

	this.getTableViewSelector = function() {
		return browser ? browser.getTableViewSelector() : "";
	};

	this.getPlayerController = function() {
		return browser ? browser.getPlayerController() : null;
	};

	this.getBrowser = function() {
		return browser;
	};

	this.removeRowAt = function(index) {
		console.log("removeRowAt " + index);
		var nsdk_path = NSDK_PROXY.pageAt(path);
		if (nsdk_path.rowsCount == undefined)
			nsdk_path.rowsCount = 0;
		else
			nsdk_path.rowsCount -= 1;
		return rows.splice(index, 1);
	}
	this.insertRowAt = function(index) {
		console.log("insertRowAt " + index);
		var nsdk_path = NSDK_PROXY.pageAt(path);
		if (nsdk_path.rowsCount == undefined)
			nsdk_path.rowsCount = 0;
		nsdk_path.rowsCount += 1;
		return self.getRows(index, nsdk_path.rowsCount - index);
	}

	this.updateRowAt = function(index) {
		console.log("updateRowAt " + index);
		var nsdk_path = NSDK_PROXY.pageAt(path);
		return self.getRows(index, 1);
	}

	this.fireEvent = function(id, context) {
		if (typeof rows[id + self.current_from] === 'undefined') {
			return;
		}

		rows[id + self.current_from].onClick(context);
	};

	this.getPath = function() {
		return path;
	};

	this.getRequestedPath = function() {
		return requested_path;
	};

	this.activate = function() {
		self.renderRows();
	};

	this.deactivate = function() {};

	this.destroy = function() {
		NSDK_PROXY.close(path);

		// Break cycle references between BrowserPage and BrowserItem
		$.each(rows, function(index, row) {
			if (row) row.destroy()
		});
		rows = [];

		// Break cycle references between Browser and BrowserPage
		browser = null;
	};

	/**
	 * Updates content-table;
	 */
	this.updateContentTable = function() {
		// The browser instance ref is needed
		if (!browser) {
			console.warn("updateContentTable: BrowserPage is destroyed, exiting");
			return;
		}

		var nsdk_path = NSDK_PROXY.pageAt(path);
		var from = self.current_from;
		var content_table = [];
		var count = $(browser.getTableViewSelector()).tableview("option", "per_page");
		var per_page = Math.min(count, nsdk_path.rowsCount + nsdk_path.extra_roles.length);
		if (rows[from]) {
			for (var i = 0; i < per_page; i++) {
				// Because getRows is asynchrous sometimes it happens that some rows
				// has not yet been resolved, and their value is null/undefined
				if (rows[i + from] !== undefined && rows[i + from] !== null) {
					$(browser.getTableViewSelector() + " .item[data-id='" + i + "']").attr('class', rows[i + from].getClassName());
					$(browser.getTableViewSelector() + " .item[data-id='" + i + "']").html(rows[i + from].getContent());
					$(browser.getTableViewSelector() + " .item[data-id='" + i + "']").css("visibility", "visible");
				} else {
					$(browser.getTableViewSelector() + " .item[data-id='" + i + "']").css("visibility", "hidden");
				}
			}
			while (i < count) {
				$(browser.getTableViewSelector() + " .item[data-id='" + i + "']").css("visibility", "hidden");
				i++;
			}
		} else {
			for (var i = from; i < count + from; i++) {
				$(browser.getTableViewSelector() + " .item[data-id='" + i + "']").css("visibility", "hidden");
			}
		}
	}

	/**
	 * renders content-table
	 */
	var timeout;
	this.renderContentTable = function(top) {
		// The browser instance ref is needed
		if (!browser) {
			console.warn("renderContentTable: BrowserPage is destroyed, exiting");
			return;
		}

		var nsdk_path = NSDK_PROXY.pageAt(path);

		// There is race condition that this method is called before it is initialized
		// this method will be called again when initialization is finished
		if (nsdk_path.initialized === false) {
			return;
		}

		var numberToFetch = 40;
		if (typeof top !== 'undefined') {
			self.current_from = top;
		}
		var from = self.current_from;
		var content_table = [];
		var per_page = $(browser.getTableViewSelector()).tableview("option", "per_page");
		var max_count = nsdk_path.rowsCount + nsdk_path.extra_roles.length;
		if (per_page > max_count) {
			per_page = max_count;
		}
		$(browser.getTableViewSelector()).tableview("option", "count", max_count);
		for (var i = from; i < per_page + from; i++) {
			if (typeof rows[i] === 'undefined') {
				if (timeout) {
					clearTimeout(timeout);
				}
				(function(i) {
					timeout = setTimeout(function() {
						var c = Math.max(nsdk_path.rowsCount, max_count);
						if (c > numberToFetch) {
							c = numberToFetch;
						}
						for (var j = i - i % numberToFetch; j < c; ++j) {
							rows.splice(j, 0, new BrowserItem(j, dummyRow, nsdk_path.rowsVersion, self));
						}
						if (nsdk_path.rowsCount < per_page + from) {
							for (var k = 0; k < nsdk_path.extra_roles.length; ++k) {
								rows.splice(nsdk_path.rowsCount + k, 0, new BrowserItem(nsdk_path.extra_roles[k], nsdk_path.get(nsdk_path.extra_roles[k]), nsdk_path.rowsVersion, self));
							}
						}
						self.getRows(i - i % numberToFetch, c).done(function() {
							self.renderContentTable()
						})
					}, 100);
				})(i);
				for (var t = i; t < per_page + from; ++t) {
					content_table.push({
						class: "item item-type-disabled",
						html: '<div class="item-icon"><img src="' + textToHtml(iconPath("skin:iconBlank")) + '"></div><div class="item-name"><span class="title unknown">unknown</span></div><div class="context none">&nbsp;</div>'
					});
				}
				break;
			} else {
				content_table.push(rows[i].render());
			}
		}

		$(browser.getTableViewSelector()).tableview("renderPage", content_table);
	}

	/**
	 * Renders current browsing page
	 */
	this.renderRows = function() {
		// The browser instance ref is needed
		if (!browser) {
			console.warn("renderRows: BrowserPage is destroyed, exiting");
			return;
		}

		var nsdk_path = NSDK_PROXY.pageAt(path);

		browser.onWorkingStateChange(false);

		var content = "";

		// Content name
		var desc = "";

		switch (nsdk_path.get("containerType")) {
			case "path":
				desc = "Directory listing";
				break;
			case "search":
				desc = "Search menu";
				break;
			case "query":
				desc = "Query result";
				break;
			case "context":
				desc = "Context menu";
				break;
			default:
				desc = "";
				break;
		}

		var ctx_name = "";
		var search = "";
		if (nsdk_path.get("search")) {
			// Construct search menu button
			$("#search").show();
		} else {
			$("#search").hide();
		}

		if (context)
			if (context.relatedRoles && context.relatedRoles.title) {
				ctx_name = " (" + context.relatedRoles.title + ")";
			}
		$("#page-title").html(textToHtml((nsdk_path.get("title") || nsdk_path.get("name") || desc) + ctx_name));
		$("#page-icon").attr("src", iconPath(nsdk_path.get("icon")));
		//content +=  "<hr/>";

		browser.onRenderRowsEvent(self);
		self.renderContentTable(0);

		$("#search").off('click').on('click', function() {
			var search = nsdk_path.get("search");
			browser.openPage(search.path, {
				'relatedRoles': context.roles
			});
		});
	};

	this.getContext = function() {
		return context;
	};

	this.onBrowseFinished = function(data) {
		// The browser instance ref is needed
		if (!browser) {
			console.warn("onBrowseFinished: BrowserPage is destroyed, exiting");
			return;
		}

		console.debug("onBrowseFinished for path: " + path + " and got data: " + JSON.stringify(data));
		// if data contains a non-empty error object set it to page_error var
		// to be used in refresh to skip getting rows
		var nsdk_path = NSDK_PROXY.pageAt(data);
		if (data.error) {
			self.page_error = data.error;
			self.page_error.path = nsdk_path.get("title");
			browser.onWorkingStateChange(false);
			browser.onErrorMessage(self.page_error);
			browser.closePage();
		} else if (path !== data) {
			path = data;
		}
		if (nsdk_path.hasOwnProperty('rowsCount')) {
			self.renderRows();
		}
	};

	this.scrollTo = function(top) {
		if (self.current_from != top) {
			self.renderContentTable(top);
		}
	}

	/**
	 *	Unlike all other  places using StreamAPI, this function expects from,count instead of from,to to define the range of rows
	 */
	this.getRows = function(from, count) {
		var ret = $.Deferred();
		if (count > 0) {
			var nsdk_path = NSDK_PROXY.pageAt(path);
			_loadMoreDataInternal(nsdk_path, ret, from, count);
			return ret.promise();
		}
		return ret.reject();
	}

	/**
	 *	_loadMoreDataInternal fetches rows starting with "from"
	 *	until "count" is reached
	 *	Unlike all other  places using StreamAPI, this function expects from,count instead of from,to to define the range of rows
	 */
	function _loadMoreDataInternal(nsdk_path, ret, from, count) {
		var to = from + count - 1;
		// clip from/to if we know the rowsCount
		if (nsdk_path.rowsCount) {
			if (from >= nsdk_path.rowsCount) {
				from = nsdk_path.rowsCount - 1;
			}
			if (to >= nsdk_path.rowsCount) {
				to = nsdk_path.rowsCount - 1;
			}
		}

		nsdk_path.rows(from, to)
			.done(function(rowsCount, newRows) {
				if (!browser) {
					console.warn("_loadMoreDataInternal nsdk_path.rows done: BrowserPage is destroyed, exiting");
					ret.reject();
					return;
				}
				if (newRows.error) {
					self.page_error = newRows.error;
					self.page_error.path = nsdk_path.get("title");
					browser.onErrorMessage(self.page_error);
					ret.reject();
				} else {
					if (rowsCount < count) {
						count = rowsCount;
					}
					// If the currently saved rowsCount value wasn't set yet, do so
					if (nsdk_path.rowsCount && nsdk_path.rowsCount != rowsCount) {
						self.page_error = "Update the page";
						self.page_error.path = nsdk_path.get("title");
						browser.onErrorMessage(self.page_error);
						ret.reject();
						return;
					}
					var length = newRows.length;
					for (var i = 0; i < length; i++) {
						rows[from + i] = new BrowserItem(from + i, newRows[i], nsdk_path.rowsVersion, self);
					}
					var limit = self.current_from;
					// checks if we are still within ["current_from", "current_from + per page"] range, the place where the request was sent
					// if yes update the content table.
					if (browser.page_stack[browser.page_stack.length - 1].getPath() == path && (from >= limit) && (from < limit + $(browser.getTableViewSelector()).tableview("option", "per_page"))) {
						self.updateContentTable();
					}
					// continue to load the rows until count is reached
					if (length != 0 && count - length > 0) {
						_loadMoreDataInternal(nsdk_path, ret, from + length, count - length);
					} else {
						ret.resolve();
					}
				}
			})
			.fail(function(error) {
				// The browser instance ref is needed
				if (!browser) {
					console.warn("_loadMoreDataInternal nsdk_path.rows fail: BrowserPage is destroyed, exiting");
					ret.reject();
					return;
				}
				self.page_error = error;
				self.page_error.path = nsdk_path.get("title");
				browser.closePage();
				browser.onErrorMessage(self.page_error);
				ret.reject();
			});
	}
}
