$(document).ready(function() {

  const defaultPage = "avr-control";

  var url = 'ws://' + window.location.hostname + ':50001/';
  window.webSocket = new Socket(url);
  window.webSocket.openCallback = function() {
    hideOverlayWarning();
  };
  window.webSocket.closeCallback = function(shouldReconnect) {
    if (shouldReconnect) {
      showSocketDisconnectError();
      setTimeout(reconnect, 2000);
    } else {
      showUnableToConnectError();
    }
  };
  window.webSocket.connect();
  window.webSocket.setCallback(function(msg) {
    if (msg[0] === 0x41 && msg[1] === 0x4D && msg[2] === 0x58) {
      parseAMX(msg);
    } else if (msg[2] === 0x33) {
      handleEngineeringMenu(msg);
    } else if (msg[2] === 0x00) {
      handlePowerOffMessage(msg[5]);
      window.socketMessageReceived(msg);
    } else if (msg[2] === 0x14) {
      handleMenuMessage(msg[5]);
    } else {
      window.socketMessageReceived(msg);
    }
  });

  function reconnect() {
    window.webSocket.connect();
  }

  function parseAMX(msg) {
    const amx = Utils.asciiFromBuffer(msg, { min: 0, max: msg.length });
    let device = amx.match(/Device-Make=(.+?)>/);
    if (device !== null) {
      setDeviceName(device[1]);
    }

    if (getDeviceName() === "JBL SYNTHESIS") {
      $(".dante-field").show();
      $("#2ch-mode option[value='8']").show();
      $("#mch-mode option[value='5']").show();
    }

    let model = amx.match(/Device-Model=(.+?)>/);
    if (model !== null) {
      setDeviceModel(model[1]);
    }

    if (getDeviceModel() === "AVR10" || getDeviceModel() === "AVR11" || getDeviceModel() === "AVR5") {
      $("#zone-settings-menu").hide();
    }

    if (getDeviceModel() === "AVR5") {
      $(".not-avr5").hide();
    }
  }

  function handleEngineeringMenu(msg) {
    localStorage.setItem("regionCode", msg[15]);
    localizeText();
    if (window.currentPage === "engineering") {
      window.socketMessageReceived(msg);
    }
  }

  function handlePowerOffMessage(status) {
    if (status === 0) {
      // showSocketDisconnectError();
    } else {
      hideOverlayWarning();
    }
  }

  function handleMenuMessage(status) {
      if (status === 0) {
        hideOverlayWarning();
      } else if (status === 2) {
        showOpenMenuWarning();
      }
  }

  /// Get device description
  window.webSocket.sendRaw([0x41, 0x4D, 0x58, 0x0D]);

  /// Check if menu open on device
  window.webSocket.send(0x14, [0xF0]);

  /// Get region from engineering menu
  window.webSocket.send(0x33, [0xF0]);

  /// Set default page
  window.currentPage = localStorage.getItem("currentPage") || defaultPage;

  $.ajaxSetup({ cache: false });

  /***************************************************************************
   * Warnings
   **************************************************************************/

  function showSocketDisconnectError() {
    const html = `
      <h1>Device disconnected.</h1>
      <p class="d-block">Please ensure the unit is still powered on and connected to the network, then refresh this page.</p>
    `;
    $("#fsol-warning>.message").html(html);
    $("#menu-open-overlay").fadeIn(100);
  }

  function showUnableToConnectError() {
    const html = `
      <h1>Unable to connect</h1>
      <p class="d-block">Please ensure IP control is enabled by:</p>
      <p class="d-block">Pressing MENU > General Setup > Control > IP</p>
    `;
    $("#fsol-warning>.message").html(html);
    $("#menu-open-overlay").fadeIn(100);
  }

  function showOpenMenuWarning() {
    const html = `
      <h1>Front panel menu open.</h1>
      <p class="d-block">Please close the front panel menu to continue.</p>
    `;
    $("#fsol-warning>.message").html(html);
    $("#menu-open-overlay").fadeIn(100);
  }

  function hideOverlayWarning() {
    $("#menu-open-overlay").fadeOut(100);
  }

  /***************************************************************************
   * Navigation
   **************************************************************************/

  function loadPage(page) {
    // Update nav active link
    $("nav li a").removeClass('active');
    $('nav li a[href*="' + page + '"]').addClass('active');

    // Always respond to stepper, unless overridden
    window.shouldRespondToStepper = function() {
      return true;
    };

    window.currentPage = page;
    localStorage.setItem("currentPage", page);
    $("main").load(page + ".html", function() {
      $.getScript("public/js/" + page + ".js", function(data, textStatus, jqhxr) {
        hideUnsupportedSpeakers();
        localizeText();
      });
    });
  }

  function hideUnsupportedSpeakers() {
    if (getDeviceModel() === "AVR10" || getDeviceModel() === "AVR11" || getDeviceModel() === "AVR5") {
      $("div.optional").hide();
      $("select#use-channels-6-7 option[value=2]").hide();
    }

    if (getDeviceModel() === "AVR5") {
      $(".not-avr5").hide();
    }
  }

  function localizeText() {
    var region = parseInt(localStorage.getItem("regionCode"), 10);
    if (region === 1 || region === 2) {
      $('body :not(script)').contents().filter(function() {
        return this.nodeType === 3;
      }).replaceWith(function() {
        return this.nodeValue
          .replace(/([Cc])entre/g, "\$1enter")
          .replace(/([Aa])nalogue/g, "\$1nalog")
          .replace(/([Dd])ialogue/g, "\$1ialog");
      });
    }
  }

  $(".nav-item a").each(function() {
    id = $(this).attr("href");
    $(this).attr('page', id.substring(1));
  });

  // Navigation handling
  $("a.nav-link").click(function(e) {
    if ($(this).attr('path') === 'root') {
      return;
    }

    e.preventDefault();

    if (window.currentPage === $(this).attr('page')) {
      return;
    }

    // Load page and update history
    loadPage($(this).attr('page'));
    history.pushState(window.currentPage, $(this).text(), null);

    return false;
  });

  // Load default page.
  loadPage(window.currentPage);

  // Add support for browser back button
  window.addEventListener('popstate', function(e) {
    var page = e.state;
    if (page == null) {
      loadPage(defaultPage);
    } else {
      loadPage(page);
    }
  });

  /***************************************************************************
   * Event Handling
   **************************************************************************/

  // Store values as they come in.
  $(document).on('input', 'input, select', function() {
    const key = $(this).attr("id");

    if (key === "source" || key === "region") {
      return;
    }

    // Don't send Speaker Distances, that should be done manually
    if (currentPage === "speaker-distances" || currentPage === "speaker-types") {
      return;
    }

    if (key === "input-name") {
      return;
    }

    const value = $(this).val();
    const model = updateModel(key, value);
    sendModel(model);
  });

  // Update range labels as the user moves the slider.
  $(document).on("input", "input[type=range]", function() {
    const type = $(this).attr("id");
    const value = $(this).val();
    updateOutputLabel(type, value);
  });
});

// Handle stepper button presses
$(document).on('click', 'button.stepper', function() {
  const id = $(this).attr('id').split('-');
  const modifier = id.shift();
  const key = id.join('-');

  if (window.shouldRespondToStepper(key, modifier) === false) {
    return;
  }

  const $slider = $('input#' + key);
  if ($slider.length === 0) {
    return;
  }

  var min = parseFloat($slider.attr("min"));
  var max = parseFloat($slider.attr("max"));
  var step = parseFloat($slider.attr("step"));

  var currentVal = parseFloat($slider.val());
  if (modifier === "increase") {
    currentVal += step;
  } else {
    currentVal -= step;
  }

  if (currentVal < min || currentVal > max) {
    return;
  }

  const model = updateModel(key, Number(currentVal));
  sendModel(model);
  updateValuesUsing(model);
});

function updateModel(key, value) {
    var model = JSON.parse(sessionStorage.getItem(window.currentPage));
    model[key] = value;
    sessionStorage.setItem(window.currentPage, JSON.stringify(model));
    return model;
}

function getValueFor(key) {
    var model = JSON.parse(sessionStorage.getItem(window.currentPage));
    return model[key];
}

function sendModel(model) {
    var encoded = window.encodeModel(model);
    window.webSocket.send(encoded.code, encoded.data);
}

/**
* Update output label for a given input type.
* @param {element} type The input element to read from.
* @param {number} value The current value.
*/
function updateOutputLabel(type, value) {
  if (type === "balance") {
    updateBalance(value);
    return;
  }

  if (type === "lip-sync") {
    updateLipSync(value);
    return;
  }

  if (type === "dolby-leveller" && value == "-1") {
    setDolbyLevellerOutputLabelToOff();
    return;
  }

  const suffix = suffixForType(type);
  $("output#" + type).val(value + suffix);
}

function setDolbyLevellerOutputLabelToOff() {
  $("output#dolby-leveller").val("Off");
}

/**
* Update balance output label.
* @param {number} input The current balance value.
*/
function updateBalance(input) {
  var value = Number(input);
  var prefix = "";

  if (value !== 0) {
    prefix = value < 0 ? "L +" : "R +";
    value = Math.abs(value);
  }

  var output = prefix + value + "dB";
  $("output#balance").val(output);
}

/**
* Update lipsync output label.
* @param {number} input The current lipsync value.
*/
function updateLipSync(input) {
  var value = Number(input) * 5;
  var output = value + "ms";
  $("output#lip-sync").val(output);
}

/**
* Returns the suffix for a given property. For example, `lip-sync` should have the suffix `ms`.
* @param {string} type The property name
*/
function suffixForType(type) {
  switch (type) {
    case "lip-sync":
    return "ms";

    case "dolby-leveller":
    case "auro-matic-strength":
    case "maximum-volume":
    case "maximum-on-volume":
    return "";

    default:
    return "dB";
  }
}

/**
* Update a pages values using a model
* @param {object} model The model to use
* @param {string} page The current page prefix
*/
function updateValuesUsing(model) {
  const oldValues = JSON.parse(sessionStorage.getItem(window.currentPage)) || {};

  for (var property in model) {
    const newValue = model[property];
    const storedValue = oldValues[property];
    var field = $("#" + property);
    // You can't update a value while it is disabled, so enable it first..
    var isDisabled = field.attr('disabled');
    if (isDisabled) {
      field.attr('disabled', false);
    }

    field.val(newValue);

    // Then disable again.
    if (isDisabled) {
      field.attr('disabled', true);
    }

    if (field.attr('type') == 'range') {
      updateOutputLabel(property, newValue);
    }
  }

  const merged = Object.assign(oldValues, model);
  sessionStorage.setItem(window.currentPage, JSON.stringify(merged));
}

/**
* Restore a pages previous values.
*/
function restorePreviousValues() {
    var model = JSON.parse(sessionStorage.getItem(window.currentPage));

    if(window.currentPage == null) {
      return;
    }

    updateValuesUsing(model);
}

/**
* Show an alert.
* @param {string} text The alert text to display
* @param {object} element The presenting button or object
* @param {number} dismissTime A timeout to dismiss the alert
* @param {string} bootstrapClass An optional bootstrap class, defaults to `.alert-primary`
*/
function showAlert(text, element, dismissTime, bootstrapClass) {
  const alertClass = "alert-" + element.attr('id');

  // Don't add duplicate alerts
  if ($("." + alertClass).length) {
    return;
  }

  if (!bootstrapClass) {
    bootstrapClass = "alert-primary";
  }

  var alertElement = document.createElement("div");
  alertElement.innerHTML = text;
  alertElement.className = "alert d-inline alert-left " + alertClass + " " + bootstrapClass;
  element.parent().append(alertElement);

  $(alertElement)
    .hide()
    .fadeIn(200)
    .delay(dismissTime)
    .fadeOut(500, function() {
      $(this).remove();
    });
}

/**
 * Convert a hexadecimal string to Uint8Array.
 * @param {string} string The hexadecimal string to convert.
 */
function hexStringToByte(string) {
  if (!string) {
    return new Uint8Array();
  }

  var a = [];

  for (var i = 0, len = string.length; i < len; i += 2) {
    a.push(parseInt(string.substr(i, 2), 16));
  }

  return new Uint8Array(a);
}

/**
 * Validate the command code within the buffer.
 * @param {Uint8Array} buffer The buffer to validate.
 * @param {number} cc The expected command code.
 */
function validateCommandCode(buffer, cc) {
  return buffer[2] == cc;
}

(function($) {
  $.fn.inputFilter = function(inputFilter) {
    return this.on("input keydown keyup mousedown mouseup select contextmenu drop", function() {
      if (inputFilter(this.value)) {
        this.oldValue = this.value;
        this.oldSelectionStart = this.selectionStart;
        this.oldSelectionEnd = this.selectionEnd;
      } else if (this.hasOwnProperty("oldValue")) {
        this.value = this.oldValue;
        this.setSelectionRange(this.oldSelectionStart, this.oldSelectionEnd);
      }
    });
  };
}(jQuery));

function setDeviceModel(model) {
  sessionStorage.setItem("deviceModel", model);
}

function getDeviceModel() {
  return sessionStorage.getItem("deviceModel");
}

function setDeviceName(name) {
  sessionStorage.setItem("deviceName", name);
}

function getDeviceName() {
  return sessionStorage.getItem("deviceName");
}
